% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/date2week.R, R/week2date.R
\name{date2week}
\alias{date2week}
\alias{week2date}
\title{Convert date to a an arbitrary week definition}
\usage{
date2week(
  x,
  week_start = get_week_start(),
  floor_day = factor,
  numeric = FALSE,
  factor = FALSE,
  ...
)

week2date(x, week_start = get_week_start(), floor_day = FALSE)
}
\arguments{
\item{x}{a \link{Date}, \link{POSIXt}, \link{character}, or any data that can be easily
converted to a date with \code{\link[=as.POSIXlt]{as.POSIXlt()}}.}

\item{week_start}{a number indicating the start of the week based on the ISO
8601 standard from 1 to 7 where 1 = Monday OR an abbreviation of the
weekdate in an English or current locale. \emph{Note: using a non-English locale
may render your code non-portable.} Defaults to the value of
\code{\link[=get_week_start]{get_week_start()}}}

\item{floor_day}{when \code{TRUE}, the days will be set to the start of the week.}

\item{numeric}{if \code{TRUE}, only the numeric week be returned. If \code{FALSE}
(default), the date in the format "YYYY-Www-d" will be returned.}

\item{factor}{if \code{TRUE}, a factor will be returned with levels spanning the
range of dates. This should only be used with \code{floor_day = TRUE} to
produce the sequence of weeks between the first and last date as the
factor levels.  Currently, \code{floor_date = FALSE} will still work, but will
produce a message indicating that it is deprecated. \emph{Take caution when
using this with a large date range as the resulting factor can contain all
days between dates}.}

\item{...}{arguments passed to \code{\link[=as.POSIXlt]{as.POSIXlt()}}, unused in all other cases.}
}
\value{
\itemize{
\item \code{date2week()} an \link[=aweek-class]{aweek} object which represents dates in
\code{YYYY-Www-d} format where \code{YYYY} is the year (associated with the week,
not necessarily the day), \code{Www} is the week number prepended by a "W" that
ranges from 01-53 and \code{d} is the day of the week from 1 to 7 where 1
represents the first day of the week (as defined by the \code{week_start}
attribute).
\item \code{week2date()} a \link{Date} object.
}
}
\description{
Convert date to a an arbitrary week definition
}
\details{
Weeks differ in their start dates depending on context. The ISO
8601 standard specifies that Monday starts the week
(\url{https://en.wikipedia.org/wiki/ISO_week_date}) while the US CDC uses
Sunday as the start of the week
(\url{https://stacks.cdc.gov/view/cdc/22305}). For
example, MSF has varying start dates depending on country in order to
better coordinate response.

While there are packages that provide conversion for ISOweeks and epiweeks,
these do not provide seamless conversion from dates to epiweeks with
non-standard start dates. This package provides a lightweight utility to
be able to convert each day.
}
\note{
\code{date2week()} will initially convert the input with \code{\link[=as.POSIXlt]{as.POSIXlt()}} and
use that to calculate the week. If the user supplies character input, it
is expected that the input will be of the format yyyy-mm-dd \emph{unless} the
user explicitly passes the "format" parameter to \code{\link[=as.POSIXlt]{as.POSIXlt()}}. If the
input is not in yyyy-mm-dd and the format parameter is not passed,
\code{date2week()} will result in an error.
}
\examples{

## Dates to weeks -----------------------------------------------------------

# The same set of days will occur in different weeks depending on the start
# date. Here we can define a week before and after today

print(dat <- as.Date("2018-12-31") + -6:7)

# By default, the weeks are defined as ISO weeks, which start on Monday
print(iso_dat <- date2week(dat))

# This can be changed by setting the global default with set_week_start()

set_week_start("Sunday")

date2week(dat)

# If you want lubridate-style numeric-only weeks, you need look no further
# than the "numeric" argument
date2week(dat, numeric = TRUE)

# To aggregate weeks, you can use `floor_day = TRUE`
date2week(dat, floor_day = TRUE)

# If you want aggregations into factors that include missing weeks, use
# `floor_day = TRUE, factor = TRUE`:
date2week(dat[c(1, 14)], floor_day = TRUE, factor = TRUE)


## Weeks to dates -----------------------------------------------------------

# The aweek class can be converted back to a date with `as.Date()`
as.Date(iso_dat)

# If you don't have an aweek class, you can use week2date(). Note that the
# week_start variable is set by the "aweek.week_start" option, which we will
# set to Monday:

set_week_start("Monday")
week2date("2019-W01-1") # 2018-12-31

# This can be overidden by the week_start argument;
week2date("2019-W01-1", week_start = "Sunday") # 2018-12-30

# If you want to convert to the first day of the week, you can use the 
# `floor_day` argument
as.Date(iso_dat, floor_day = TRUE)

## The same two week timespan starting on different days --------------------
# ISO week definition: Monday -- 1
date2week(dat, 1)
date2week(dat, "Monday")

# Tuesday -- 2
date2week(dat, 2)
date2week(dat, "Tuesday")

# Wednesday -- 3
date2week(dat, 3)
date2week(dat, "W") # you can use valid abbreviations

# Thursday -- 4
date2week(dat, 4)
date2week(dat, "Thursday")

# Friday -- 5
date2week(dat, 5)
date2week(dat, "Friday")

# Saturday -- 6
date2week(dat, 6)
date2week(dat, "Saturday")

# Epiweek definition: Sunday -- 7 
date2week(dat, 7)
date2week(dat, "Sunday")
}
\seealso{
\code{\link[=set_week_start]{set_week_start()}}, \code{\link[=as.Date.aweek]{as.Date.aweek()}}, \code{\link[=print.aweek]{print.aweek()}}, \code{\link[=as.aweek]{as.aweek()}},
\code{\link[=get_aweek]{get_aweek()}}
}
\author{
Zhian N. Kamvar
}
