# ifndef _RHEO_TRACE_H
# define _RHEO_TRACE_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

/*Class:trace
NAME:  @code{trace} - restriction to boundary values operator 
@cindex trace operator
@clindex trace
@clindex form
@clindex space
@clindex field
DESCRIPTION:
  The @code{trace} operator restricts a field to its values located to a 
  boundary value domain.
  The @code{trace} class is a derived class of the @code{form} class
  (@pxref{form class}).
  Thus, all operations, such as linear algebra, that applies
  to @code{form}, applies also to @code{trace}.
EXAMPLE:
  The following example shows how to plot the 
  trace of a field, from standard input:
  @example
    int main() @{
	geo omega ("square");
	space Vh (omega "P1");
	space Wh (omega, "P1", omega ["top"]);
	trace gamma (Vh, Wh);

	field uh;
	cin >> uh;
	field wh (Wh);
	wh = gamma*uh;
	cout << wh;
    @}
  @end example
AUTHOR: Pierre.Saramito@imag.fr
DATE:   11 dec 1998
METHODS: @trace
End:
*/

// ============================================================================
//  Includes
// ============================================================================

#include "rheolef/skit.h"
#include "rheolef/csr.h"
#include "rheolef/asr.h"
#include "rheolef/form.h"
#include "rheolef/form_diag.h"
namespace rheolef { 

// ============================================================================
//  Class "trace"
// ============================================================================

//<trace:  
class trace : public form
{
public :

//  allocator/deallocator:
  
  trace();
  trace(const space& V, const space& W);

// accessors:
  
  const space& get_space() const;
  const space& get_boundary_space() const;
  const geo&   get_geo() const;
  const geo&   get_boundary_geo() const;
};
//>trace:

// ============================================================================
//  inline'd
// ============================================================================

// _X = space on full mesh
// _Y = space on domain mesh
inline
trace::trace ()
 : form()
{
}
inline
const space& 
trace::get_space() const
{
  return get_first_space();
}
inline
const space& 
trace::get_boundary_space() const
{
  return get_second_space() ;
}
inline
const geo&
trace::get_geo() const 
{
  return get_first_space().get_geo() ;
}
inline
const geo&
trace::get_boundary_geo() const 
{
  return get_second_space().get_geo() ;
}
}// namespace rheolef
# endif /* _RHEO_TRACE_H */
