% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reprex.R
\name{reprex}
\alias{reprex}
\title{Render a reprex}
\usage{
reprex(x = NULL, input = NULL, outfile = NULL, venue = c("gh",
  "so", "ds", "r", "rtf", "html"), render = TRUE, advertise = NULL,
  si = opt(FALSE), style = opt(FALSE), show = opt(TRUE),
  comment = opt("#>"), tidyverse_quiet = opt(TRUE),
  std_out_err = opt(FALSE))
}
\arguments{
\item{x}{An expression. If not given, \code{reprex()} looks for code in
\code{input} or on the clipboard, in that order.}

\item{input}{Character. If has length one and lacks a terminating newline,
interpreted as the path to a file containing reprex code. Otherwise,
assumed to hold reprex code as character vector.}

\item{outfile}{Optional basename for output files. When \code{NULL}
(default), reprex writes to temp files below the session temp directory. If
\code{outfile = "foo"}, expect output files in current working directory,
like \code{foo_reprex.R}, \code{foo_reprex.md}, and, if \code{venue = "r"},
\code{foo_rendered.R}. If \code{outfile = NA}, expect output files in
a location and with basename derived from \code{input}, if sensible, or in
current working directory with basename derived from \code{\link[=tempfile]{tempfile()}}
otherwise.}

\item{venue}{Character. Must be one of the following (case insensitive):
\itemize{
\item "gh" for \href{https://github.github.com/gfm/}{GitHub-Flavored Markdown}, the
default
\item "so" for \href{https://stackoverflow.com/editing-help}{Stack Overflow Markdown}
\item "ds" for Discourse, e.g.,
\href{https://community.rstudio.com}{community.rstudio.com}. Note: this is
currently just an alias for "gh"!
\item "r" for a runnable R script, with commented output interleaved
\item "rtf" for \href{https://en.wikipedia.org/wiki/Rich_Text_Format}{Rich Text Format} (not supported for
un-reprexing)
\item "html" for an HTML fragment suitable for inclusion in a larger HTML
document (not supported for un-reprexing)
}}

\item{render}{Logical. Whether to call \code{\link[rmarkdown:render]{rmarkdown::render()}} on the templated
reprex, i.e. whether to actually run the code. Defaults to \code{TRUE}. Exists
primarily for the sake of internal testing.}

\item{advertise}{Logical. Whether to include a footer that describes when and
how the reprex was created. If unspecified, the option \code{reprex.advertise}
is consulted and, if that is not defined, default is \code{TRUE} for venues
\code{"gh"}, \code{"so"}, \code{"ds"}, \code{"html"} and \code{FALSE} for \code{"r"} and \code{"rtf"}.}

\item{si}{Logical. Whether to include \code{\link[devtools:session_info]{devtools::session_info()}}, if
available, or \code{\link[=sessionInfo]{sessionInfo()}} at the end of the reprex. When \code{venue} is
"gh" or "ds", the session info is wrapped in a collapsible details tag.
Read more about \code{\link[=opt]{opt()}}.}

\item{style}{Logical. Whether to style code with \code{\link[styler:style_text]{styler::style_text()}}.
Read more about \code{\link[=opt]{opt()}}.}

\item{show}{Logical. Whether to show rendered output in a viewer (RStudio or
browser). Read more about \code{\link[=opt]{opt()}}.}

\item{comment}{Character. Prefix with which to comment out output, defaults
to \code{"#>"}. Read more about \code{\link[=opt]{opt()}}.}

\item{tidyverse_quiet}{Logical. Sets the option \code{tidyverse.quiet}, which
suppresses (\code{TRUE}, the default) or includes (\code{FALSE}) the startup message
for the tidyverse package. Read more about \code{\link[=opt]{opt()}}.}

\item{std_out_err}{Logical. Whether to append a section for output sent to
stdout and stderr by the reprex rendering process. This can be necessary to
reveal output if the reprex spawns child processes or \code{system()} calls.
Note this cannot be properly interleaved with output from the main R
process, nor is there any guarantee that the lines from standard output and
standard error are in correct chronological order. See \code{\link[callr:r_safe]{callr::r_safe()}}
for more. Read more about \code{\link[=opt]{opt()}}.}
}
\value{
Character vector of rendered reprex, invisibly.
}
\description{
Run a bit of R code using \code{\link[rmarkdown:render]{rmarkdown::render()}} and write the rendered result
to user's clipboard. The goal is to make it easy to share a small
reproducible example ("reprex"), e.g., in a GitHub issue. Reprex source can
be
\itemize{
\item read from clipboard
\item read from current selection or active document in RStudio
(with \code{\link[=reprex_addin]{reprex_addin()}})
\item provided directly as expression, character vector, or string
\item read from file
}

reprex can also be used for syntax highlighting (with or without rendering);
see below for more.
}
\section{Details}{

The usual "code + commented output" is returned invisibly, put on the
clipboard, and written to file. An HTML preview displays in RStudio's Viewer
pane, if available, or in the default browser, otherwise. Leading \code{"> "}
prompts, are stripped from the input code. Read more at
\url{https://reprex.tidyverse.org/}.

reprex sets specific \href{http://yihui.name/knitr/options/}{knitr options}:
\itemize{
\item Chunk options default to \code{collapse = TRUE}, \code{comment = "#>"},
\code{error = TRUE}. Note that \code{error = TRUE}, because a common use case is bug
reporting.
\item reprex also sets knitr's \code{upload.fun}. It defaults to
\code{\link[knitr:imgur_upload]{knitr::imgur_upload()}} so figures produced by the reprex appear properly
on GitHub, Stack Overflow, or Discourse. Note that this function requires
the packages httr & xml2 or RCurl & XML, depending on your knitr version.
When \code{venue = "r"}, \code{upload.fun} is set to \code{identity}, so that figures
remain local. In that case, you may also want to set \code{outfile}.
You can supplement or override these options with special comments in your
code (see examples).
}
}

\section{Syntax highlighting}{

A secondary use case for reprex is to produce syntax highlighted code
snippets, with or without rendering, to paste into presentation software,
such as Keynote or PowerPoint. Use \code{venue = "rtf"}.

This feature is experimental and currently rather limited. It probably only
works on macOS and requires the installation of the
\href{http://www.andre-simon.de/doku/highlight/en/highlight.php}{highlight}
command line tool, which can be installed via
\href{https://formulae.brew.sh/formula/highlight}{homebrew}. This venue is
discussed in \href{https://reprex.tidyverse.org/articles/articles/rtf.html}{an article}
}

\examples{
\dontrun{
# put some code like this on the clipboard
# (y <- 1:4)
# mean(y)
reprex()

# provide code as an expression
reprex(rbinom(3, size = 10, prob = 0.5))
reprex({y <- 1:4; mean(y)})
reprex({y <- 1:4; mean(y)}, style = TRUE)

# note that you can include newlines in those brackets
# in fact, that is often a good idea
reprex({
  x <- 1:4
  y <- 2:5
  x + y
})

## provide code via character vector
reprex(input = c("x <- 1:4", "y <- 2:5", "x + y"))

## if just one line, terminate with '\\n'
reprex(input = "rnorm(3)\\n")

## customize the output comment prefix
reprex(rbinom(3, size = 10, prob = 0.5), comment = "#;-)")

# override a default chunk option
reprex({
  #+ setup, include = FALSE
  knitr::opts_chunk$set(collapse = FALSE)

  #+ actual-reprex-code
  (y <- 1:4)
  median(y)
})

# add prose, use general markdown formatting
reprex({
  #' # A Big Heading
  #'
  #' Look at my cute example. I love the
  #' [reprex](https://github.com/tidyverse/reprex#readme) package!
  y <- 1:4
  mean(y)
}, advertise = FALSE)

# read reprex from file
tmp <- file.path(tempdir(), "foofy.R")
writeLines(c("x <- 1:4", "mean(x)"), tmp)
reprex(input = tmp)

# read from file and write to similarly-named outfiles
reprex(input = tmp, outfile = NA)
list.files(dirname(tmp), pattern = "foofy")

# clean up
file.remove(list.files(dirname(tmp), pattern = "foofy", full.names = TRUE))

# write rendered reprex to file
tmp <- file.path(tempdir(), "foofy")
reprex({
  x <- 1:4
  y <- 2:5
  x + y
}, outfile = tmp)
list.files(dirname(tmp), pattern = "foofy")

# clean up
file.remove(list.files(dirname(tmp), pattern = "foofy", full.names = TRUE))

# write reprex to file AND keep figure local too, i.e. don't post to imgur
tmp <- file.path(tempdir(), "foofy")
reprex({
  #+ setup, include = FALSE
  knitr::opts_knit$set(upload.fun = identity)

  #+ actual-reprex-code
  #' Some prose
  ## regular comment
  (x <- 1:4)
  median(x)
  plot(x)
  }, outfile = tmp)
list.files(dirname(tmp), pattern = "foofy")

# clean up
unlink(
  list.files(dirname(tmp), pattern = "foofy", full.names = TRUE),
  recursive = TRUE
)

## target venue = Stack Overflow
## https://stackoverflow.com/editing-help
ret <- reprex({
  x <- 1:4
  y <- 2:5
  x + y
}, venue = "so")
ret

## target venue = R, also good for email or Slack snippets
ret <- reprex({
  x <- 1:4
  y <- 2:5
  x + y
}, venue = "R")
ret

## target venue = html
ret <- reprex({
  x <- 1:4
  y <- 2:5
  x + y
}, venue = "html")
ret

## include prompt and don't comment the output
## use this when you want to make your code hard to execute :)
reprex({
  #+ setup, include = FALSE
  knitr::opts_chunk$set(comment = NA, prompt = TRUE)

  #+ actual-reprex-code
  x <- 1:4
  y <- 2:5
  x + y
})

## leading prompts are stripped from source
reprex(input = c("> x <- 1:3", "> median(x)"))
}

}
