% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fonts.R
\name{font_google}
\alias{font_google}
\alias{font_link}
\alias{font_face}
\alias{font_collection}
\alias{is_font_collection}
\title{Helpers for importing web fonts}
\usage{
font_google(
  family,
  local = TRUE,
  cache = sass_file_cache(sass_cache_context_dir()),
  wght = NULL,
  ital = NULL,
  display = c("swap", "auto", "block", "fallback", "optional")
)

font_link(family, href)

font_face(
  family,
  src,
  weight = NULL,
  style = NULL,
  display = c("swap", "auto", "block", "fallback", "optional"),
  stretch = NULL,
  variant = NULL,
  unicode_range = NULL
)

font_collection(..., default_flag = TRUE, quote = TRUE)

is_font_collection(x)
}
\arguments{
\item{family}{A character string with a \emph{single} font family name.}

\item{local}{Whether or not download and bundle local (woff) font files.}

\item{cache}{A \code{\link[=sass_file_cache]{sass_file_cache()}} object (or, more generally, a file
caching class with \verb{$get_file()} and \verb{$set_file()} methods). Set this
argument to \code{FALSE} or \code{NULL} to disable caching.}

\item{wght}{One of the following:
\itemize{
\item \code{NULL}, the default weight for the \code{family}.
\item A character string defining an \href{https://developers.google.com/fonts/docs/css2#axis_ranges}{axis range}
\item A numeric vector of desired font weight(s).
}}

\item{ital}{One of the following:
\itemize{
\item \code{NULL}, the default \code{font-style} for the \code{family}.
\item \code{0}, meaning \code{font-style: normal}
\item \code{1}, meaning \code{font-style: italic}
\item \code{c(0, 1)}, meaning both \code{normal} and \code{italic}
}}

\item{display}{A character vector for the \code{font-display} \verb{@font-face}
property.}

\item{href}{A URL resource pointing to the font data.}

\item{src}{A character vector for the \code{src} \verb{@font-face} property. Beware
that is character strings are taken verbatim, so careful quoting and/or URL
encoding may be required.}

\item{weight}{A character (or numeric) vector for the \code{font-weight}
\verb{@font-face} property.}

\item{style}{A character vector for the \code{font-style} \verb{@font-face} property.}

\item{stretch}{A character vector for the \code{font-stretch} \verb{@font-face}
property.}

\item{variant}{A character vector for the \code{font-variant} \verb{@font-face}
property.}

\item{unicode_range}{A character vector for \code{unicode-range} \verb{@font-face}
property.}

\item{...}{a collection of \code{font_google()}, \code{font_link()}, \code{font_face()}, and/or character vector(s) (i.e., family names to include in the CSS \code{font-family} properly). Family names are automatically quoted as necessary.}

\item{default_flag}{whether or not to include a \code{!default} when converted to a Sass variable with \code{\link[=as_sass]{as_sass()}}.}

\item{quote}{whether or not to attempt automatic quoting of family names.}

\item{x}{test whether \code{x} is a \code{font_collection()}, \code{font_google()}, \code{font_link()}, or \code{font_face()} object.}
}
\value{
a \code{\link[=sass_layer]{sass_layer()}} holding an \code{\link[htmltools:htmlDependency]{htmltools::htmlDependency()}} which points
to the font files.
}
\description{
Include font file(s) when defining a Sass variable that represents a CSS
\code{font-family} property.
}
\details{
These helpers \strong{must be used the named list approach to variable
definitions}, for example:

\if{html}{\out{<div class="sourceCode">}}\preformatted{list(
  list("font-variable" = font_google("Pacifico")),
  list("body\{font-family: $font-variable\}")
)
}\if{html}{\out{</div>}}
}
\section{Font fallbacks}{


By default, \code{font_google()} downloads, caches, and serves the relevant font
file(s) locally. By locally serving files, there's a guarantee that the font
can render in any client browser, even when the client doesn't have internet
access. However, when importing font files remotely (i.e., \code{font_google(..., local = FALSE)} or \code{font_link()}), it's a good idea to provide fallback
font(s) in case the remote link isn't working (e.g., maybe the end user
doesn't have an internet connection). To provide fallback fonts, use
\code{\link[=font_collection]{font_collection()}}, for example:

\if{html}{\out{<div class="sourceCode">}}\preformatted{pacifico <- font_google("Pacifico", local = FALSE)
as_sass(list(
  list("font-variable" = font_collection(pacifico, "system-ui")),
  list("body\{font-family: $font-variable\}")
))
}\if{html}{\out{</div>}}
}

\section{Default flags}{


These font helpers encourage best practice of adding a \code{!default} to Sass
variable definitions, but the flag may be removed via \code{font_collection()} if
desired.

\if{html}{\out{<div class="sourceCode">}}\preformatted{as_sass(list("font-variable" = pacifico))
#> $font-variable: Pacifico !default;
as_sass(list("font-variable" = font_collection(pacifico, default_flag = F)))
#> $font-variable: Pacifico;
}\if{html}{\out{</div>}}
}

\section{Serving non-Google fonts locally}{


Non-Google fonts may also be served locally with \code{font_face()}, but it
requires downloading font file(s) and pointing \code{src} to the right location
on disk. If you want \code{src} to be a relative file path (you almost certainly
do), then you'll need to mount that resource path using something like
\code{\link[shiny:resourcePaths]{shiny::addResourcePath()}} (for a shiny app) or \code{servr::httd()} (for static
HTML).
}

\examples{

library(htmltools)

my_font <- list("my-font" = font_google("Pacifico"))
hello <- tags$body(
  "Hello",
  tags$style(
    sass(
      list(
        my_font,
        list("body {font-family: $my-font}")
      )
    )
  )
)

if (interactive()) {
  browsable(hello)
}

# Three different yet equivalent ways of importing a remotely-hosted Google Font
a <- font_google("Crimson Pro", wght = "200..900", local = FALSE)
b <- font_link(
  "Crimson Pro",
  href = "https://fonts.googleapis.com/css2?family=Crimson+Pro:wght@200..900"
)
url <- "https://fonts.gstatic.com/s/crimsonpro/v13/q5uDsoa5M_tv7IihmnkabARboYF6CsKj.woff2"
c <- font_face(
  family = "Crimson Pro",
  style = "normal",
  weight = "200 900",
  src = paste0("url(", url, ") format('woff2')")
)
}
\references{
\url{https://developers.google.com/fonts/docs/css2}

\url{https://developer.mozilla.org/en-US/docs/Web/CSS/@font-face}

\url{https://developer.mozilla.org/en-US/docs/Learn/CSS/Styling_text/Web_fonts}
}
