% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rectangle.R
\name{hoist}
\alias{hoist}
\alias{unnest_longer}
\alias{unnest_wider}
\alias{unnest_auto}
\title{Rectangle a nested list into a tidy tibble}
\usage{
hoist(
  .data,
  .col,
  ...,
  .remove = TRUE,
  .simplify = TRUE,
  .ptype = NULL,
  .transform = NULL
)

unnest_longer(
  data,
  col,
  values_to = NULL,
  indices_to = NULL,
  indices_include = NULL,
  names_repair = "check_unique",
  simplify = TRUE,
  ptype = NULL,
  transform = NULL
)

unnest_wider(
  data,
  col,
  names_sep = NULL,
  simplify = TRUE,
  strict = FALSE,
  names_repair = "check_unique",
  ptype = NULL,
  transform = NULL
)

unnest_auto(data, col)
}
\arguments{
\item{.data, data}{A data frame.}

\item{.col, col}{List-column to extract components from.

For \code{hoist()} and \code{unnest_auto()}, this must identify a single column.

For \code{unnest_wider()} and \code{unnest_longer()}, you can use tidyselect to
select multiple columns to unnest simultaneously. When using
\code{unnest_longer()} with multiple columns, values across columns that
originated from the same row are recycled to a common size.}

\item{...}{Components of \code{.col} to turn into columns in the form
\code{col_name = "pluck_specification"}. You can pluck by name with a character
vector, by position with an integer vector, or with a combination of the
two with a list. See \code{\link[purrr:pluck]{purrr::pluck()}} for details.

The column names must be unique in a call to \code{hoist()}, although existing
columns with the same name will be overwritten. When plucking with a
single string you can choose to omit the name, i.e. \code{hoist(df, col, "x")}
is short-hand for \code{hoist(df, col, x = "x")}.}

\item{.remove}{If \code{TRUE}, the default, will remove extracted components
from \code{.col}. This ensures that each value lives only in one place. If all
components are removed from \code{.col}, then \code{.col} will be removed from the
result entirely.}

\item{.simplify, simplify}{If \code{TRUE}, will attempt to simplify lists of
length-1 vectors to an atomic vector. Can also be a named list containing
\code{TRUE} or \code{FALSE} declaring whether or not to attempt to simplify a
particular column. If a named list is provided, the default for any
unspecified columns is \code{TRUE}.}

\item{.ptype, ptype}{Optionally, a named list of prototypes declaring the
desired output type of each component. Alternatively, a single empty
prototype can be supplied, which will be applied to all components. Use
this argument if you want to check that each element has the type you
expect when simplifying.

If a \code{ptype} has been specified, but \code{simplify = FALSE} or simplification
isn't possible, then a \link[vctrs:list_of]{list-of} column will be returned
and each element will have type \code{ptype}.}

\item{.transform, transform}{Optionally, a named list of transformation
functions applied to each component. Alternatively, a single function can
be supplied, which will be applied to all components. Use this argument if
you want to transform or parse individual elements as they are extracted.

When both \code{ptype} and \code{transform} are supplied, the \code{transform} is applied
before the \code{ptype}.}

\item{values_to}{A string giving the column name (or names) to store the
unnested values in. If multiple columns are specified in \code{col}, this can
also be a glue string containing \code{"{col}"} to provide a template for the
column names. The default, \code{NULL}, gives the output columns the same names
as the input columns.}

\item{indices_to}{A string giving the column name (or names) to store the
the inner names or positions (if not named) of the values. If multiple
columns are specified in \code{col}, this can also be a glue string containing
\code{"{col}"} to provide a template for the column names. The default, \code{NULL},
gives the output columns the same names as \code{values_to}, but suffixed with
\code{"_id"}.}

\item{indices_include}{A single logical value specifying whether or not to
add an index column. If any value has inner names, the index column will be
a character vector of those names, otherwise it will be an integer vector
of positions. If \code{NULL}, defaults to \code{TRUE} if any value has inner names
or if \code{indices_to} is provided.

If \code{indices_to} is provided, then \code{indices_include} must not be \code{FALSE}.}

\item{names_repair}{Used to check that output data frame has valid
names. Must be one of the following options:
\itemize{
\item "minimal": no name repair or checks, beyond basic existence,
\item "unique": make sure names are unique and not empty,
\item "check_unique": (the default), no name repair, but check they are unique,
\item "universal": make the names unique and syntactic
\item a function: apply custom name repair.
\item \link{tidyr_legacy}: use the name repair from tidyr 0.8.
\item a formula: a purrr-style anonymous function (see \code{\link[rlang:as_function]{rlang::as_function()}})
}

See \code{\link[vctrs:vec_as_names]{vctrs::vec_as_names()}} for more details on these terms and the
strategies used to enforce them.}

\item{names_sep}{If \code{NULL}, the default, the names will be left
as is. If a string, the outer and inner names will be pasted together using
\code{names_sep} as a separator.

If the values being unnested are unnamed and \code{names_sep} is supplied, the
inner names will be automatically generated as an increasing sequence of
integers.}

\item{strict}{A single logical specifying whether or not to apply strict
vctrs typing rules. If \code{FALSE}, typed empty values (like \code{list()} or
\code{integer()}) nested within list-columns will be treated like \code{NULL} and
will not contribute to the type of the unnested column. This is useful
when working with JSON, where empty values tend to lose their type
information and show up as \code{list()}.}
}
\description{
\code{hoist()}, \code{unnest_longer()}, and \code{unnest_wider()} provide tools for
rectangling, collapsing deeply nested lists into regular columns.
\code{hoist()} allows you to selectively pull components of a list-column out
in to their own top-level columns, using the same syntax as \code{\link[purrr:pluck]{purrr::pluck()}}.
\code{unnest_wider()} turns each element of a list-column into a column, and
\code{unnest_longer()} turns each element of a list-column into a row.
\code{unnest_auto()} picks between \code{unnest_wider()} or \code{unnest_longer()}
based on heuristics described below.

Learn more in \code{vignette("rectangle")}.
}
\section{Unnest variants}{


The three \code{unnest()} functions differ in how they change the shape of the
output data frame:
\itemize{
\item \code{unnest_wider()} preserves the rows, but changes the columns.
\item \code{unnest_longer()} preserves the columns, but changes the rows
\item \code{\link[=unnest]{unnest()}} can change both rows and columns.
}

These principles guide their behaviour when they are called with a
non-primary data type. For example, if you \code{unnest_wider()} a list of data
frames, the number of rows must be preserved, so each column is turned into
a list column of length one. Or if you \code{unnest_longer()} a list of data
frames, the number of columns must be preserved so it creates a packed
column. I'm not sure how if these behaviours are useful in practice, but
they are theoretically pleasing.
}

\section{\code{unnest_auto()} heuristics}{

\code{unnest_auto()} inspects the inner names of the list-col:
\itemize{
\item If all elements are unnamed, it uses
\code{unnest_longer(indices_include = FALSE)}.
\item If all elements are named, and there's at least one name in
common across all components, it uses \code{unnest_wider()}.
\item Otherwise, it falls back to \code{unnest_longer(indices_include = TRUE)}.
}
}

\examples{
df <- tibble(
  character = c("Toothless", "Dory"),
  metadata = list(
    list(
      species = "dragon",
      color = "black",
      films = c(
        "How to Train Your Dragon",
        "How to Train Your Dragon 2",
        "How to Train Your Dragon: The Hidden World"
       )
    ),
    list(
      species = "blue tang",
      color = "blue",
      films = c("Finding Nemo", "Finding Dory")
    )
  )
)
df

# Turn all components of metadata into columns
df \%>\% unnest_wider(metadata)

# Choose not to simplify list-cols of length-1 elements
df \%>\% unnest_wider(metadata, simplify = FALSE)
df \%>\% unnest_wider(metadata, simplify = list(color = FALSE))

# Extract only specified components
df \%>\% hoist(metadata,
  "species",
  first_film = list("films", 1L),
  third_film = list("films", 3L)
)

df \%>\%
  unnest_wider(metadata) \%>\%
  unnest_longer(films)

# unnest_longer() is useful when each component of the list should
# form a row
df <- tibble(
  x = 1:3,
  y = list(NULL, 1:3, 4:5)
)
df \%>\% unnest_longer(y)
# Automatically creates names if widening
df \%>\% unnest_wider(y)
# But you'll usually want to provide names_sep:
df \%>\% unnest_wider(y, names_sep = "_")

# And similarly if the vectors are named
df <- tibble(
  x = 1:2,
  y = list(c(a = 1, b = 2), c(a = 10, b = 11, c = 12))
)
df \%>\% unnest_wider(y)
df \%>\% unnest_longer(y)

# Both unnest_wider() and unnest_longer() allow you to unnest multiple
# columns at once. This is particularly useful with unnest_longer(), where
# unnesting sequentially would generate a cartesian product of the rows.
df <- tibble(
  x = 1:2,
  y = list(1:2, 3:4),
  z = list(5:6, 7:8)
)
unnest_longer(df, c(y, z))
unnest_longer(unnest_longer(df, y), z)

# With JSON, it is common for empty elements to be represented by `list()`
# rather then their typed equivalent, like `integer()`
json <- list(
  list(x = 1:2, y = 1:2),
  list(x = list(), y = 3:4),
  list(x = 3L, y = list())
)
df <- tibble(json = json)

# The defaults of `unnest_wider()` treat empty types (like `list()`) as `NULL`.
# This chains nicely into `unnest_longer()`.
wide <- unnest_wider(df, json)
wide

unnest_longer(wide, c(x, y))

# To instead enforce strict vctrs typing rules, use `strict`
wide_strict <- unnest_wider(df, json, strict = TRUE)
wide_strict

try(unnest_longer(wide_strict, c(x, y)))
}
\seealso{
For complex inputs where you need to rectangle a nested list according
to a specification, see the
\href{https://CRAN.R-project.org/package=tibblify}{tibblify} package.
}
