/*
 *	recite - english text speech synthesizer
 *	Copyright (C) 1993 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 1, or (at your option)
 *	any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to translate frames into waveforms
 *
 * Portions of this file are derived floatfrom a program
 * Copyright 1982 by Dennis H. Klatt:
 * Klatt synthesizer
 * Modified version of synthesizer described in
 * J. Acoust. Soc. Am., Mar. 1980. -- new voicing source.
 */

#include <math.h>

#include <error.h>
#include <parwave.h>
#include <r250.h>
#include <trace.h>


/*
 * variables to hold speaker definition from host
 */
static long outsl;	/* Output waveform selector */
static long samrate;	/* Number of output samples per second */
static long FLPhz;	/* Frequeny of glottal downsample low-pass filter */
static long BLPhz;	/* Bandwidt of glottal downsample low-pass filter */
static long nfcascade; /* Number of formants in cascade vocal tract    */
static long glsource;  /* 1->impulsive, 2->natural voicing source      */

/*
 * variables to hold input parameters from host
 */
static long F0hz10;  /* Voicing fund freq in Hz		       */
static long F1hz  ;  /* First formant freq in Hz,  200 to 1300      */
static long F2hz  ;  /* Second formant freq in Hz,  550 to 3000     */
static long F3hz  ;  /* Third formant freq in Hz, 1200 to 4999      */
static long F4hz  ;  /* Fourth formant freq in Hz, 1200 to 4999     */
static long F5hz  ;  /* Fifth formant freq in Hz, 1200 to 4999      */
static long F6hz  ;  /* Sixth formant freq in Hz, 1200 to 4999      */
static long FNZhz ;  /*    Nasal zero freq in Hz,  248 to  528      */
static long FNPhz ;  /*    Nasal pole freq in Hz,  248 to  528      */

static long B1hz  ;  /*   First formant bw in Hz,   40 to 1000      */
static long B2hz  ;  /*  Second formant bw in Hz,   40 to 1000      */
static long B3hz  ;  /*   Third formant bw in Hz,   40 to 1000      */
static long B4hz  ;  /*   Fourth formant bw in Hz,  40 to 1000      */
static long B5hz  ;  /*   Fifth formant bw in Hz,   40 to 1000      */
static long B6hz  ;  /*   Sixth formant bw in Hz,   40 to 2000      */
static long B1phz ;  /* Par. 1st formant bw in Hz,   40 to 1000     */
static long B2phz ;  /* Par. 2nd formant bw in Hz,   40 to 1000     */
static long B3phz ;  /* Par. 3rd formant bw in Hz,   40 to 1000     */
static long B4phz ;  /* Par. 4th formant bw in Hz,  40 to 1000      */
static long B5phz ;  /* Par. 5th formant bw in Hz,   40 to 1000     */
static long B6phz ;  /* Par. 6th formant bw in Hz,   40 to 2000     */
static long BNZhz ;  /*      Nasal zero bw in Hz,   40 to 1000      */
static long BNPhz ;  /*      Nasal pole bw in Hz,   40 to 1000      */

static long AVdb  ;  /*     Amp of voicing in dB,    0 to   70      */
static long AVpdb ;  /* Amp of voicing, par in dB,    0 to   70     */
static long AP   ;  /*   Amp of aspiration in dB,    0 to   70      */
static long AF   ;  /*    Amp of frication in dB,    0 to   80      */
static long A1   ;  /* Amp of par 1st formant in dB, 0 to   80      */
static long ANP  ;  /* Amp of par nasal pole in dB,  0 to   80      */
static long A2   ;  /* Amp of F2 frication in dB,    0 to   80      */
static long A3   ;  /* Amp of F3 frication in dB,    0 to   80      */
static long A4   ;  /* Amp of F4 frication in dB,    0 to   80      */
static long A5   ;  /* Amp of F5 frication in dB,    0 to   80      */
static long A6   ;  /* Amp of F6 (same as r6pa),     0 to   80      */
static long AB   ;  /* Amp of bypass fric. in dB,    0 to   80      */
static long TLTdb ;  /* Voicing spectral tilt in dB,  0 to   24     */
static long Kopen ;  /* # of samples in open period, 10 to   65     */
static long Aturb ;  /* Breathiness in voicing,      0 to   80      */
static long Kskew ;  /* Skewness of alternate periods,0 to   40     */
		    /* in sample#/2				 */
static long Gain0 ;  /* Overall gain, 60 dB is unity  0 to   60     */

/*
 * same variables converted to linear floating point
 */
static double amp_parF1; /* A1 converted to linear gain	      */
static double amp_parFNP; /* AP converted to linear gain	     */
static double amp_parF2; /* A2 converted to linear gain	      */
static double amp_parF3; /* A3 converted to linear gain	      */
static double amp_parF4; /* A4 converted to linear gain	      */
static double amp_parF5; /* A5 converted to linear gain	      */
static double amp_parF6; /* A6 converted to linear gain	      */
static double amp_bypas; /* AB converted to linear gain	      */
static double amp_voice; /* AVdb converted to linear gain	    */
static double par_amp_voice; /* AVpdb converted to linear gain       */
static double amp_aspir; /* AP converted to linear gain	      */
static double amp_frica; /* AF converted to linear gain	      */
static double amp_breth; /* ATURB converted to linear gain	   */
static double amp_gain0; /* G0 converted to linear gain	      */

/* COUNTERS */

static long ns   ;  /* Number of samples into current frame	 */
static long nper  ;  /* Current loc in voicing period   40000 samp/s */
static long n4   ;  /* Counter of 4 samples in glottal source loop  */

/* COUNTER LIMITS */

static long T0   ;  /* Fundamental period in output samples times 4 */
static long nopen ;  /* Number of samples in open phase of period    */
static long nmod  ;  /* Position in period to begin noise amp. modul */
static long nspfr ;  /* Number of samples in a parameter frame      */

/* ALL-PURPOSE TEMPORARY VARIABLES */

static long temp    ;
static double temp1 ;

/* VARIABLES THAT HAVE TO STICK AROUND FOR AWHILE, AND THUS "temp" */
/* IS NOT APPROPRIATE */

static long nrand   ;  /* Varible used by random number generator      */
static long skew    ;  /* Alternating jitter, in half-period units     */

static double a     ;  /* Makes waveshape of glottal pulse when open   */
static double b     ;  /* Makes waveshape of glottal pulse when open   */
static double voice ;  /* Current sample of voicing waveform	   */
static double vwave ;  /* Ditto, but before multiplication by AVdb     */
static double noise ;  /* Output of random number generator	    */
static double frics ;  /* Frication sound source		       */
static double aspiration; /* Aspiration sound source		   */
static double sourc ;  /* Sound source if all-parallel config used     */
static double casc_next_in;  /* Input to next used resonator of casc   */
static double out   ;  /* Output of cascade branch, also final output  */
static double rnzout;  /* Output of cascade nazal zero resonator       */
static double glotout; /* Output of glottal sound source	       */
static double par_glotout; /* Output of parallelglottal sound sourc    */
static double outbypas; /* Output signal from bypass path	      */

/* INTERNAL MEMORY FOR DIGITAL RESONATORS AND ANTIRESONATOR	   */

static double rnpp_1;  /* Last output sample from parallel nasal pole  */
static double rnpp_2;  /* Second-previous output sample		*/

static double r1p_1 ;  /* Last output sample from parallel 1st formant */
static double r1p_2 ;  /* Second-previous output sample		*/

static double r2p_1 ;  /* Last output sample from parallel 2nd formant */
static double r2p_2 ;  /* Second-previous output sample		*/

static double r3p_1 ;  /* Last output sample from parallel 3rd formant */
static double r3p_2 ;  /* Second-previous output sample		*/

static double r4p_1 ;  /* Last output sample from parallel 4th formant */
static double r4p_2 ;  /* Second-previous output sample		*/

static double r5p_1 ;  /* Last output sample from parallel 5th formant */
static double r5p_2 ;  /* Second-previous output sample		*/

static double r6p_1 ;  /* Last output sample from parallel 6th formant */
static double r6p_2 ;  /* Second-previous output sample		*/

static double r1c_1 ;  /* Last output sample from cascade 1st formant  */
static double r1c_2 ;  /* Second-previous output sample		*/

static double r2c_1 ;  /* Last output sample from cascade 2nd formant  */
static double r2c_2 ;  /* Second-previous output sample		*/

static double r3c_1 ;  /* Last output sample from cascade 3rd formant  */
static double r3c_2 ;  /* Second-previous output sample		*/

static double r4c_1 ;  /* Last output sample from cascade 4th formant  */
static double r4c_2 ;  /* Second-previous output sample		*/

static double r5c_1 ;  /* Last output sample from cascade 5th formant  */
static double r5c_2 ;  /* Second-previous output sample		*/

static double r6c_1 ;  /* Last output sample from cascade 6th formant  */
static double r6c_2 ;  /* Second-previous output sample		*/

static double rnpc_1;  /* Last output sample from cascade nasal pole   */
static double rnpc_2;  /* Second-previous output sample		*/

static double rnz_1 ;  /* Last output sample from cascade nasal zero   */
static double rnz_2 ;  /* Second-previous output sample		*/

static double rgl_1 ;  /* Last output crit-damped glot low-pass filter */
static double rgl_2 ;  /* Second-previous output sample		*/

static double rlp_1 ;  /* Last output from downsamp low-pass filter    */
static double rlp_2 ;  /* Second-previous output sample		*/

static double rout_1 ;  /* Last output sample from output low-pass  */
static double rout_2 ;  /* Second-previous output sample		*/

static double vlast ;  /* Previous output of voice		     */
static double nlast ;  /* Previous output of random number generator   */
static double glotlast; /* Previous value of glotout		   */

/* COEFFICIENTS FOR DIGITAL RESONATORS AND ANTIRESONATOR */

static double rnppa ;   /* "a" coefficient for parallel nasal pole     */
static double rnppb ;   /* "b" coefficient			     */
static double rnppc ;   /* "c" coefficient			     */

static double r1pa  ;  /* "a" coef for par. 1st formant		*/
static double r1pb  ;  /* "b" coefficient			      */
static double r1pc  ;  /* "c" coefficient			      */

static double r2pa  ;  /* Could be same as A2 if all integer impl.     */
static double r2pb  ;  /* "b" coefficient			      */
static double r2pc  ;  /* "c" coefficient			      */

static double r3pa  ;  /* Could be same as A3 if all integer impl.     */
static double r3pb  ;  /* "b" coefficient			      */
static double r3pc  ;  /* "c" coefficient			      */

static double r4pa  ;  /* Could be same as A4 if all integer impl.     */
static double r4pb  ;  /* "b" coefficient			      */
static double r4pc  ;  /* "c" coefficient			      */

static double r5pa  ;  /* Could be same as A5 if all integer impl.     */
static double r5pb  ;  /* "b" coefficient			      */
static double r5pc  ;  /* "c" coefficient			      */

static double r6pa  ;  /* Could be same as A6 if all integer impl.     */
static double r6pb  ;  /* "b" coefficient			      */
static double r6pc  ;  /* "c" coefficient			      */

static double r1ca  ;  /* Could be same as r1pa if all integer impl.   */
static double r1cb  ;  /* Could be same as r1pb if all integer impl.   */
static double r1cc  ;  /* Could be same as r1pc if all integer impl.   */

static double r2ca  ;   /* "a" coefficient for cascade 2nd formant     */
static double r2cb  ;   /* "b" coefficient			     */
static double r2cc  ;   /* "c" coefficient			     */

static double r3ca  ;   /* "a" coefficient for cascade 3rd formant     */
static double r3cb  ;   /* "b" coefficient			     */
static double r3cc  ;   /* "c" coefficient			     */

static double r4ca  ;   /* "a" coefficient for cascade 4th formant     */
static double r4cb  ;   /* "b" coefficient			     */
static double r4cc  ;   /* "c" coefficient (same as R4Cccoef)	  */

static double r5ca  ;   /* "a" coefficient for cascade 5th formant     */
static double r5cb  ;   /* "b" coefficient			     */
static double r5cc  ;   /* "c" coefficient (same as R5Cccoef)	  */

static double r6ca  ;   /* "a" coefficient for cascade 5th formant     */
static double r6cb  ;   /* "b" coefficient			     */
static double r6cc  ;   /* "c" coefficient (same as R5Cccoef)	  */

static double rnpca ;   /* "a" coefficient for cascade nasal pole      */
static double rnpcb ;   /* "b" coefficient			     */
static double rnpcc ;   /* "c" coefficient			     */

static double rnza  ;   /* "a" coefficient for cascade nasal zero      */
static double rnzb  ;   /* "b" coefficient			     */
static double rnzc  ;   /* "c" coefficient			     */

static double rgla  ;   /* "a" coefficient for crit-damp glot filter   */
static double rglb  ;   /* "b" coefficient			     */
static double rglc  ;   /* "c" coefficient			     */

static double rlpa  ;   /* "a" coefficient for downsam low-pass filter */
static double rlpb  ;   /* "b" coefficient			     */
static double rlpc  ;   /* "c" coefficient			     */

static double routa  ;   /* "a" coefficient for output low-pass filter */
static double routb  ;   /* "b" coefficient			     */
static double routc  ;   /* "c" coefficient			     */

static double decay ;   /* TLTdb converted to exponential time const   */
static double onemd ;   /* in voicing one-pole low-pass filter	 */


/* CONSTANTS AND TABLES TO BE PUT IN ROM			      */

#define CASCADE_PARALLEL      1 /* Normal synthesizer config	  */
#define ALL_PARALLEL	  2 /* Only use parallel branch	   */


/*
 * Constant B0 controls shape of glottal pulse as a function
 * of desired duration of open phase N0
 * (Note that N0 is specified in terms of 40,000 samples/sec of speech)
 *
 *    Assume voicing waveform V(t) has form: k1 t**2 - k2 t**3
 *
 *    If the radiation characterivative, a temporal derivative
 *      is folded in, and we go from continuous time to discrete
 *      integers n:  dV/dt = vwave[n]
 *			 = sum over i=1,2,...,n of { a - (i * b) }
 *			 = a n  -  b/2 n**2
 *
 *      where the  constants a and b control the detailed shape
 *      and amplitude of the voicing waveform over the open
 *      potion of the voicing cycle "nopen".
 *
 *    Let integral of dV/dt have no net dc flow --> a = (b * nopen) / 3
 *
 *    Let maximum of dUg(n)/dn be constant --> b = gain / (nopen * nopen)
 *      meaning as nopen gets bigger, V has bigger peak proportional to n
 *
 *    Thus, to generate the table below for 40 <= nopen <= 263:
 *
 *      B0[nopen - 40] = 1920000 / (nopen * nopen)
 */
short B0[224] = {
	1200,   1142,   1088,   1038,   991,
	948,    907,    869,    833,    799,
	768,    738,    710,    683,    658,
	634,    612,    590,    570,    551,
	533,    515,    499,    483,    468,
	454,    440,    427,    415,    403,
	391,    380,    370,    360,    350,
	341,    332,    323,    315,    307,
	300,    292,    285,    278,    272,
	265,    259,    253,    247,    242,
	237,    231,    226,    221,    217,
	212,    208,    204,    199,    195,
	192,    188,    184,    180,    177,
	174,    170,    167,    164,    161,
	158,    155,    153,    150,    147,
	145,    142,    140,    137,    135,
	133,    131,    128,    126,    124,
	122,    120,    119,    117,    115,
	113,    111,    110,    108,    106,
	105,    103,    102,    100,    99,
	97,     96,     95,     93,     92,
	91,     90,     88,     87,     86,
	85,     84,     83,     82,     80,
	79,     78,     77,     76,     75,
	75,     74,     73,     72,     71,
	70,     69,     68,     68,     67,
	66,     65,     64,     64,     63,
	62,     61,     61,     60,     59,
	59,     58,     57,     57,     56,
	56,     55,     55,     54,     54,
	53,     53,     52,     52,     51,
	51,     50,     50,     49,     49,
	48,     48,     47,     47,     46,
	46,     45,     45,     44,     44,
	43,     43,     42,     42,     41,
	41,     41,     41,     40,     40,
	39,     39,     38,     38,     38,
	38,     37,     37,     36,     36,
	36,     36,     35,     35,     35,
	35,     34,     34,     33,     33,
	33,     33,     32,     32,     32,
	32,     31,     31,     31,     31,
	30,     30,     30,     30,     29,
	29,     29,     29,     28,     28,
	28,     28,     27,     27
};

/*
 * Convertion table, db to linear, 87 dB --> 32767
 *				 86 dB --> 29491 (1 dB down = 0.5**1/6)
 *				 ...
 *				 81 dB --> 16384 (6 dB down = 0.5)
 *				 ...
 *				  0 dB -->     0
 *
 * The just noticeable difference for a change in intensity of a vowel
 *   is approximately 1 dB.  Thus all amplitudes are quantized to 1 dB
 *   steps.
 */

static double amptable[88] =
{
	    0.,     0.,     0.,     0.,     0.,
	    0.,     0.,     0.,     0.,     0.,
	    0.,     0.,     0.,     6.,     7.,
	    8.,     9.,    10.,    11.,    13.,
	   14.,    16.,    18.,    20.,    22.,
	   25.,    28.,    32.,    35.,    40.,
	   45.,    51.,    57.,    64.,    71.,
	   80.,    90.,   101.,   114.,   128.,
	  142.,   159.,   179.,   202.,   227.,
	  256.,   284.,   318.,   359.,   405.,
	  455.,   512.,   568.,   638.,   719.,
	  811.,   911.,  1024.,  1137.,  1276.,
	 1438.,  1622.,  1823.,  2048.,  2273.,
	 2552.,  2875.,  3244.,  3645.,  4096.,
	 4547.,  5104.,  5751.,  6488.,  7291.,
	 8192.,  9093., 10207., 11502., 12976.,
	14582., 16384., 18350., 20644., 23429.,
	26214., 29491., 32767
};

spkrdef_ty spkrdef =
{
	0,		/* OUTS */
	SAMP_RATE,	/* Srat */
	NSAMP_PER_FRAME, /* NSfr */
	900,		/* Flp */
	600,		/* BWlp */
	3,		/* NFca */
	NATURAL	/* VSsw */
};

static	long	sigmx;
static	long	dispt;
static	long	disptcum;
static	long	ms25;
static	double	r8ca;
static	double	r8cb;
static	double	r8cc;
static	double	r7ca;
static	double	r7cb;
static	double	r7cc;
static	double	r8c_1;
static	double	r8c_2;
static	double	r7c_1;
static	double	r7c_2;
static	int	count = 0;
static	double	minus_pi_t;
static	double	two_pi_t;



/*
 * NAME
 *	setabc
 *
 * SYNOPSIS
 *	void setabc(long f, long bw, double *a, double *b, double *c);
 *
 * DESCRIPTION
 *	The setabc function is used to
 *      convert formant freqencies and bandwidth into
 *      resonator difference equation constants
 */

static void setabc _((long f, long bw, double *a, double *b, double *c));

static void
setabc(f, bw, acoef, bcoef, ccoef)
	long	f;		/* Frequency of resonator in Hz */
	long	bw;		/* Bandwidth of resonator in Hz */
	double	*acoef;
	double	*bcoef;
	double	*ccoef;
{
	double	r;
	double	arg;

	/*
	 * Let r =  exp(-pi bw t)
	 */
	arg = minus_pi_t * bw;
	r = exp(arg);

	/*
	 * Let c =  -r**2
	 */
	*ccoef = -(r * r);

	/*
	 * Let b = r * 2*cos(2 pi f t)
	 */
	arg = two_pi_t * f;
	*bcoef = r * cos(arg) * 2.;

	/*
	 * Let a = 1.0 - b - c
	 */
	*acoef = 1.0 - *bcoef - *ccoef;

	trace(("f=%ld bw=%ld acoef=%8.5f bcoef=%8.5f ccoef=%8.5f\n",
		f, bw, *acoef, *bcoef, *ccoef));
}


static void resonclr _((void));

static void
resonclr()
{
	 rnpp_1 = 0;  /* Last output sample from parallel nasal pole  */
	 rnpp_2 = 0;  /* Second-previous output sample		*/

	 r1p_1 = 0;  /* Last output sample from parallel 1st formant */
	 r1p_2 = 0;  /* Second-previous output sample		*/

	 r2p_1 = 0;  /* Last output sample from parallel 2nd formant */
	 r2p_2 = 0;  /* Second-previous output sample		*/

	 r3p_1 = 0;  /* Last output sample from parallel 3rd formant */
	 r3p_2 = 0;  /* Second-previous output sample		*/

	 r4p_1 = 0;  /* Last output sample from parallel 4th formant */
	 r4p_2 = 0;  /* Second-previous output sample		*/

	 r5p_1 = 0;  /* Last output sample from parallel 5th formant */
	 r5p_2 = 0;  /* Second-previous output sample		*/

	 r6p_1 = 0;  /* Last output sample from parallel 6th formant */
	 r6p_2 = 0;  /* Second-previous output sample		*/

	 r1c_1 = 0;  /* Last output sample from cascade 1st formant  */
	 r1c_2 = 0;  /* Second-previous output sample		*/

	 r2c_1 = 0;  /* Last output sample from cascade 2nd formant  */
	 r2c_2 = 0;  /* Second-previous output sample		*/

	 r3c_1 = 0;  /* Last output sample from cascade 3rd formant  */
	 r3c_2 = 0;  /* Second-previous output sample		*/

	 r4c_1 = 0;  /* Last output sample from cascade 4th formant  */
	 r4c_2 = 0;  /* Second-previous output sample		*/

	 r5c_1 = 0;  /* Last output sample from cascade 5th formant  */
	 r5c_2 = 0;  /* Second-previous output sample		*/

	 r6c_1 = 0;  /* Last output sample from cascade 6th formant  */
	 r6c_2 = 0;  /* Second-previous output sample		*/

	 r7c_1 = 0;
	 r7c_2 = 0;

	 r8c_1 = 0;
	 r8c_2 = 0;
  
	 rnpc_1 = 0;  /* Last output sample from cascade nasal pole   */
	 rnpc_2 = 0;  /* Second-previous output sample		*/

	 rnz_1 = 0;  /* Last output sample from cascade nasal zero   */
	 rnz_2 = 0;  /* Second-previous output sample		*/

	 rgl_1 = 0;  /* Last output crit-damped glot low-pass filter */
	 rgl_2 = 0;  /* Second-previous output sample		*/

	 rlp_1 = 0;  /* Last output from downsamp low-pass filter    */
	 rlp_2 = 0;  /* Second-previous output sample		*/

	 vlast = 0;  /* Previous output of voice		     */
	 nlast = 0;  /* Previous output of random number generator   */
	 glotlast = 0; /* Previous value of glotout		   */
}


/*
 * NAME
 *	DBtoLIN
 *
 * SYNOPSIS
 *	double DBtoLIN(long dB);
 *
 * DESCRIPTION
 *	The DBtoLIN function is used to
 *      convert from decibels to a linear scale factor
 */

static double DBtoLIN _((long dB));

static double
DBtoLIN(dB)
	long	dB;
{
	double	lgtemp;

	/*
	 * Check limits or argument (can be removed in final product)
	 */
	if (dB < 0)
	{
		trace(("Try to compute amptable[%d]\n", dB));
		return(0);
	}		
	lgtemp = amptable[dB] * .001;
	return (lgtemp);
}


/*
 * NAME
 *	setzeroabc
 *
 * SYNOPSIS
 *	void setzeroabc(long f, long bw, double *a, double *b, double *c);
 *
 * DESCRIPTION
 *	The setzeroabc function is used to
 *	convert formant freqencies and bandwidth into
 *      anti-resonator difference equation constants
 */

static void setzeroabc _((long f, long bw, double *a, double *b, double *c));

static void
setzeroabc(f, bw, acoef, bcoef, ccoef)
	long	f;		/* Frequency of resonator in Hz	 */
	long	bw;		/* Bandwidth of resonator in Hz	 */
	double	*acoef;
	double	*bcoef;
	double	*ccoef;
{
	double	r;
	double	arg;

	/*
	 * First compute ordinary resonator coefficients
	 */

	/*
	 * Let r =  exp(-pi bw t)
	 */
	arg = minus_pi_t * bw;
	r = exp(arg);

	/*
	 * Let c =  -r**2
	 */
	*ccoef = -(r * r);

	/*
	 * Let b = r * 2*cos(2 pi f t)
	 */
	arg = two_pi_t * f;
	*bcoef = r * cos(arg) * 2.;

	/*
	 * Let a = 1.0 - b - c
	 */
	*acoef = 1. - *bcoef - *ccoef;

	/*
	 * Now convert to antiresonator coefficients (a'=1/a, b'=b/a, c'=c/a)
	 */
	*acoef = 1.0 / *acoef;
	*ccoef *= -*acoef;
	*bcoef *= -*acoef;

	trace(("fz=%3d bw=%3d acoef=%8.5f bcoef=%8.5f ccoef=%8.5f\n",
		f, bw, *acoef, *bcoef, *ccoef));
}


/*
 * NAME
 *	gethost
 *
 * SYNOPSIS
 *	void gethost(void);
 *
 * DESCRIPTION
 *	The gethost function is used to
 *	get the variable parameters from host computer,
 *	initially also get definition of fixed pars
 */

static void gethost _((pars_ty *));

static void
gethost(pars)
	pars_ty		*pars;
{
	static int	initsw;

	/*
	 * Initialize speaker definition
	 */
	if (!initsw)
	{
		initsw++;
   
		/*
		 * Read Speaker Definition from Host
		 */
		/* Select which waveform to output */
		outsl = spkrdef.OUTs;

		/* Sampling rate */
		samrate = SAMP_RATE;
		/* spkrdef.Srat */

		/* number of samples per frame of pars  */
		nspfr = NSAMP_PER_FRAME;
		/* spkrdef.NSfr */

		/* Freq of glottal low-pass filter */
		FLPhz = (950 * samrate) / 10000;
		/* spkrdef.Flp */

		/* Bw of glottal low-pass filter   */
		BLPhz = (630 * samrate) / 10000;
		/* spkrdef.BWlp */

		/* Number formants in cascade tract*/
		nfcascade = spkrdef.NFca;

		/* 1->impulsive, 2->natural voicing*/
		glsource = spkrdef.VSsw;

		minus_pi_t = -(4 * atan(1.)) / samrate;
		two_pi_t = -2. * minus_pi_t;
		setabc(FLPhz, BLPhz, &rlpa, &rlpb, &rlpc);

		/* Init ran # generator */
		r250_init();

		/* Number samps in 25 ms */
		ms25 = (25 * samrate) / 1000;

		/* initialize resonantors for new utterance  */
		resonclr();
		nper = 0;
		T0 = 0;
	}

	/*
	 * Read speech frame definition into temp store
	 * and move some parameters into active use immediately
	 * (voice-excited ones are updated pitch synchronously
	 * to avoid waveform glitches).
	 */
	F0hz10 = pars->F0hz10;
	AVdb = pars->AVdb - 7;
	if (AVdb < 0)   
		AVdb = 0;

	F1hz = pars->F1hz;
	B1hz = pars->B1hz;


	F2hz = pars->F2hz;
	B2hz = pars->B2hz;

	F3hz = pars->F3hz;
	B3hz = pars->B3hz;

	F4hz = pars->F4hz;
	B4hz = pars->B4hz;

	F5hz = pars->F5hz;
	B5hz = pars->B5hz;


	F6hz = pars->F6hz;	/* f  of parallel 6th formant */
	B6hz = pars->B6hz;	/* bw of cascade 6th formant (doesn't exist) */

	FNZhz = pars->FNZhz;
	BNZhz = pars->BNZhz;

	FNPhz = pars->FNPhz;
	BNPhz = pars->BNPhz;

	AP = pars->AP;
	amp_aspir = DBtoLIN(AP) * .05;
	Kopen = pars->Kopen;

	Aturb = pars->Aturb;
	TLTdb = pars->TLTdb;

	AF = pars->AF;
	amp_frica = DBtoLIN(AF) * 0.25;
	Kskew = pars->Kskew;
	AVpdb = pars->AVpdb;
	par_amp_voice = DBtoLIN(AVpdb);

	A1 = pars->A1;
	amp_parF1 = DBtoLIN(A1) * 0.4;
	B1phz = pars->B1phz;

	A2 = pars->A2;
	amp_parF2 = DBtoLIN(A2) * 0.15;
	B2phz = pars->B2phz;

	A3 = pars->A3;
	amp_parF3 = DBtoLIN(A3) * 0.06;
	B3phz = pars->B3phz;

	A4 = pars->A4;
	amp_parF4 = DBtoLIN(A4) * 0.04;
	B4phz = pars->B4phz;

	A5 = pars->A5;
	amp_parF5 = DBtoLIN(A5) * 0.022;
	B5phz = pars->B4phz;

	A6 = pars->A6;
	amp_parF6 = DBtoLIN(A6) * 0.03;
	B6phz = pars->B6phz;

	ANP = pars->ANP;
	amp_parFNP = DBtoLIN(ANP) * 0.6;

	AB = pars->AB;
	amp_bypas = DBtoLIN(AB) * 0.05;

	Gain0 = pars->Gain0 - 3;
	if (Gain0 <= 0)
		Gain0 = 57;
	amp_gain0 = DBtoLIN(Gain0);
     
#if 0
	pr_pars();
#endif

	/*
	 * Set coefficients of variable cascade resonators
	 */
	if (nfcascade >= 8)
		setabc(7500, 600, &r8ca, &r8cb, &r8cc);
	if (nfcascade >= 7)
		setabc(6500, 500, &r7ca, &r7cb, &r7cc);
	if (nfcascade >= 6)
		setabc(F6hz, B6hz, &r6ca, &r6cb, &r6cc);
	if (nfcascade >= 5)
		setabc(F5hz, B5hz, &r5ca, &r5cb, &r5cc);
	setabc(F4hz, B4hz, &r4ca, &r4cb, &r4cc);
	setabc(F3hz, B3hz, &r3ca, &r3cb, &r3cc);
	setabc(F2hz, B2hz, &r2ca, &r2cb, &r2cc);
	setabc(F1hz, B1hz, &r1ca, &r1cb, &r1cc);

	/*
	 * Set coeficients of nasal resonator and zero antiresonator
	 */
	setabc(FNPhz, BNPhz, &rnpca, &rnpcb, &rnpcc);
	setzeroabc(FNZhz, BNZhz, &rnza, &rnzb, &rnzc);

	/*
	 * Set coefficients of parallel resonators, and amplitude of outputs
	 */
	setabc(F1hz, B1hz, &r1pa, &r1pb, &r1pc);
	r1pa *= amp_parF1;
	setabc(FNPhz, BNPhz, &rnppa, &rnppb, &rnppc);
	rnppa *= amp_parFNP;
	setabc(F2hz, B2phz, &r2pa, &r2pb, &r2pc);
	r2pa *= amp_parF2;
	setabc(F3hz, B3phz, &r3pa, &r3pb, &r3pc);
	r3pa *= amp_parF3;
	setabc(F4hz, B4phz, &r4pa, &r4pb, &r4pc);
	r4pa *= amp_parF4;
	setabc(F5hz, B5phz, &r5pa, &r5pb, &r5pc);
	r5pa *= amp_parF5;
	setabc(F6hz, B6phz, &r6pa, &r6pb, &r6pc);
	r6pa *= amp_parF6;

	/*
	 * output low-pass filter
	 */
	setabc((long)0.0, (long)8000.0, &routa, &routb, &routc);

	/*
	 * Print a dot on screen every 25 ms to indicate program not dead
	 */
	dispt += nspfr;
	disptcum += nspfr;
}


/*
 * NAME
 *	resonglot
 *
 * SYNOPSIS
 *	void resonglot(void);
 *
 * DESCRIPTION
 *	The resonglot funtion is used to
 *	critically-damped Low-Pass Resonator of Impulsive Glottal Source
 */

static void resonglot _((void));

static void
resonglot()
{
	long	temp3;
	long	temp4;

	temp4 = rglc * rgl_2;	  /*   (ccoef * old2)     */
	rgl_2 = rgl_1;

	temp3 = rglb * rgl_1;	  /* + (bcoef * old1)     */
	temp4 += temp3;

	temp3 = rgla * vwave;	  /* + (acoef * input)    */
	rgl_1 = temp4 + temp3;
}


/*
 * NAME
 *	impulsive_source
 *
 * SYNOPSIS
 *	double impulsive_source(void);
 *
 * DESCRIPTION
 *	The impulsive_source function is used to
 */

static double impulsive_source _((void));

static double
impulsive_source()
{
	static double doublet[] = { 0., 13000000., -13000000. };
	
	if (nper < SIZEOF(doublet))
		vwave = doublet[nper];
	else
		vwave = 0.;

	/*
	 * Low-pass filter the differenciated impulse with a critically-damped
	 * second-order filter, time constant proportional to Kopen
	 */
	resonglot();
	return (rgl_1);
}



/*
 * NAME
 *	natural_source
 *
 * SYNOPSIS
 *	double natural_source(void);
 *
 * DESCRIPTION
 *	The natural_source function is used to
 *	Vwave is the differentiated glottal flow waveform, there is a weak
 *	spectral zero around 800 Hz, magic constants a, b reset pitch-synch
 */

static double natural_source _((void));

static double
natural_source()
{
	double	lgtemp;

	/*
	 * See if glottis open
	 */
	if (nper < nopen)
	{
		a -= b;
		vwave += a;
		lgtemp = vwave * 0.03;
		return lgtemp;
	}

	/*
	 * Glottis closed
	 */
	vwave = 0;
	return 0;
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*								     */
/*  S U B R O U T I N E   P I T C H _ S Y N C _ P A R _ U P D A T E    */
/*								     */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Reset selected parameters pitch-synchronously */

static void pitch_synch_par_reset _((void));

static void
pitch_synch_par_reset()
{
	long	zero;

	if (F0hz10 > 0)
	{
		/* Period in samp*4 */
		T0 = (40 * samrate) / F0hz10;
		amp_voice = DBtoLIN(AVdb);

		/*
		 * Duration of period before amplitude modulation
		 */
		nmod = T0;
		if (AVdb > 0)
		{
	       		nmod >>= 1;
		}

		/*
		 * Breathiness of voicing waveform
		 */
		amp_breth = DBtoLIN(Aturb) * 0.1;

		/*
		 * Set open phase of glottal period
		 * where  40 <= open phase <= 263
		 */
		nopen = 4 * Kopen;
		if ((glsource == IMPULSIVE) && (nopen > 263))
			nopen = 263;

		if (nopen >= (T0-1))
		{
			nopen = T0 - 2;
			trace(("Warning: glottal open period cannot exceed T0, truncated\n"));
		}

		if (nopen < 40)
		{
			/* F0 max = 1000 Hz */
			nopen = 40;
			trace(("Warning: minimum glottal open period is 10 samples, truncated, nopen = %d\n", nopen));
		}

		/*
		 * Reset a & b, which determine shape of
		 * "natural" glottal waveform
		 */
		b = B0[nopen - 40];
		a = (b * nopen) * .333;

		/*
		 * Reset width of "impulsive" glottal pulse
		 */
		temp = samrate / nopen;	     /* WAS 11000 !!! */
		zero = 0;
		setabc(zero, temp, &rgla, &rglb, &rglc);

		/*
		 * Make gain at F1 about constant
		 */
		temp1 = nopen *.00833;
		rgla *= temp1 * temp1;

		/*
		 * Truncate skewness so as not to exceed
		 * duration of closed phase of glottal period
		 */
		temp = T0 - nopen;
		if (Kskew > temp)
		{
			trace(("Kskew duration=%d > glottal closed period=%d, truncate\n", Kskew, T0-nopen));
			Kskew = temp;
		}
		if (skew >= 0)
		{
			/* Reset skew to requested Kskew */
			skew = Kskew;
		}
		else
		{
			skew = -Kskew;
		}

		/*
		 * Add skewness to closed portion of voicing period
		 */
		T0 = T0 + skew;
		skew = - skew;
	}
	else
	{
		/* Default for f0 undefined */
		T0 = 4;
		amp_voice = 0.;
		nmod = T0;
		amp_breth = 0.;
		a = 0.;
		b = 0.;
	}

	/*
	 * Reset these pars pitch synchronously or at update rate if f0=0
	 */
	if ((T0 != 4) || (ns == 0))
	{
		/*
		 * Set one-pole low-pass filter that tilts glottal source
		 */
		decay = (.033 * TLTdb);
		if (decay > 0.)
		{
			onemd = 1. - decay;
		}
		else
		{
			onemd = 1.;
		}
	}
}




/*
 * NAME
 *	gen_noise
 *
 * SYNOPSIS
 *	void gen_noise(void);
 *
 * DESCRIPTION
 *	The gen_noise function is used to generate a
 *	random number in the range -8192 to 8191.
 *
 * OUTPUT
 *	may be found in "nrand" unfiltered, or "noise" low pass filtered.
 */

static void gen_noise _((void));

static void
gen_noise() 
{
	nrand = (r250() & ((1L << 14) - 1)) - (1L << 13);

	/*
	 * Tilt down noise spectrum by soft low-pass filter having
	 * a pole near the origin in the z-plane, i.e.
	 * output = input + (0.75 * lastoutput)
	 */
	noise = nrand + (0.75 * nlast);
	nlast = noise;
}


/* 
 * Low-Pass Downsampling Resonator of Glottal Source
 */

static void resonlp _((void));

static void
resonlp()
{
	long	temp3;
	long	temp4;

	temp4 = rlpc * rlp_2;	  /*   (ccoef * old2)     */
	rlp_2 = rlp_1;

	temp3 = rlpb * rlp_1;	  /* + (bcoef * old1)     */
	temp4 += temp3;

	temp3 = rlpa * voice;	  /* + (acoef * input)    */
	rlp_1 = temp4 + temp3;
	voice = rlp_1;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*								     */
/*		S U B R O U T I N E   R E S O N C		    */
/*								     */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Nasal Antiresonator of Cascade Vocal Tract:
 *  Output = (rnza * input) + (rnzb * oldin1) + (rnzc * oldin2) */

static void resoncnz _((void));

static void
resoncnz()
{
	long	temp3;
	long	temp4;

	temp4 = rnzc * rnz_2;	  /*   (ccoef * oldin2)   */
	rnz_2 = rnz_1;

	temp3 = rnzb * rnz_1;	  /* + (bcoef * oldin1)   */
	temp4 += temp3;

	temp3 = rnza * glotout;	    /* + (acoef * input)    */
	rnz_1 = glotout;
	rnzout = temp4 + temp3;
}

/* Nasal Resonator of Cascade Vocal Tract */

static void resoncnp _((void));

static void
resoncnp()
{

	long temp3, temp4;

	temp4 = rnpcc * rnpc_2;	/*   (ccoef * old2)     */
	rnpc_2 = rnpc_1;

	temp3 = rnpcb * rnpc_1;	  /* + (bcoef * old1)     */
	temp4 += temp3;

	temp3 = rnpca * rnzout;	  /* + (acoef * input)    */
	rnpc_1 = temp4 + temp3;
}

/* Eighth cascaded Formant */

static void resonc8 _((void));

static void
resonc8()
{

	long temp3, temp4;

	temp4 = r8cc * r8c_2;	  /*   (ccoef * old2)     */
	r8c_2 = r8c_1;

	temp3 = r8cb * r8c_1;	  /* + (bcoef * old1)     */
	temp4 += temp3;

	temp3 = r8ca * casc_next_in;   /* + (acoef * input)    */
	r8c_1 = temp4 + temp3;
}

/* Seventh cascaded Formant */

static void resonc7 _((void));

static void
resonc7()
{

	long temp3, temp4;

	temp4 = r7cc * r7c_2;	  /*   (ccoef * old2)     */
	r7c_2 = r7c_1;

	temp3 = r7cb * r7c_1;	  /* + (bcoef * old1)     */
	temp4 += temp3;

	temp3 = r7ca * casc_next_in;   /* + (acoef * input)    */
	r7c_1 = temp4 + temp3;
}

/* Sixth cascaded Formant */

static void resonc6 _((void));

static void
resonc6()
{

	long temp3, temp4;

	temp4 = r6cc * r6c_2;	  /*   (ccoef * old2)     */
	r6c_2 = r6c_1;

	temp3 = r6cb * r6c_1;	  /* + (bcoef * old1)     */
	temp4 += temp3;

	temp3 = r6ca * casc_next_in;   /* + (acoef * input)    */
	r6c_1 = temp4 + temp3;
}

/* Fifth Formant */

static void resonc5 _((void));

static void
resonc5()
{

	long temp3, temp4;

	temp4 = r5cc * r5c_2;	  /*   (ccoef * old2)     */
	r5c_2 = r5c_1;

	temp3 = r5cb * r5c_1;	  /* + (bcoef * old1)     */
	temp4 += temp3;

	temp3 = r5ca * casc_next_in;   /* + (acoef * input)    */
	r5c_1 = temp4 + temp3;
}

/* Fourth Formant */

static void resonc4 _((void));

static void
resonc4()
{

	long temp3, temp4;

	temp4 = r4cc * r4c_2;	  /*   (ccoef * old2)     */
	r4c_2 = r4c_1;

	temp3 = r4cb * r4c_1;	  /* + (bcoef * old1)     */
	temp4 += temp3;

	temp3 = r4ca * casc_next_in;   /* + (acoef * input)    */
	r4c_1 = temp4 + temp3;
}

/* Third Formant */

static void resonc3 _((void));

static void
resonc3()
{

	long temp3, temp4;

	temp4 = r3cc * r3c_2;	  /*   (ccoef * old2)     */
	r3c_2 = r3c_1;

	temp3 = r3cb * r3c_1;	  /* + (bcoef * old1)     */
	temp4 += temp3;

	temp3 = r3ca * casc_next_in;   /* + (acoef * input)    */
	r3c_1 = temp4 + temp3;
}

/* Second Formant */

static void resonc2 _((void));

static void
resonc2()
{

	long temp3, temp4;

	temp4 = r2cc * r2c_2;	  /*   (ccoef * old2)     */
	r2c_2 = r2c_1;

	temp3 = r2cb * r2c_1;	  /* + (bcoef * old1)     */
	temp4 += temp3;

	temp3 = r2ca * casc_next_in;   /* + (acoef * input)    */
	r2c_1 = temp4 + temp3;
}

/* First Formant of Cascade Vocal Tract */

static void resonc1 _((void));

static void
resonc1()
{

	long temp3, temp4;

	temp4 = r1cc * r1c_2;	  /*   (ccoef * old2)     */
	r1c_2 = r1c_1;

	temp3 = r1cb * r1c_1;	  /* + (bcoef * old1)     */
	temp4 += temp3;

	temp3 = r1ca * casc_next_in;   /* + (acoef * input)    */
	r1c_1 = temp4 + temp3;
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*								     */
/*		  S U B R O U T I N E   R E S O N P		  */
/*								     */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*   Output = (acoef * input) + (bcoef * old1) + (ccoef * old2); */

/* Sixth Formant of Parallel Vocal Tract */

static void reson6p _((void));

static void
reson6p()
{

	long temp3, temp4;

	temp4 = r6pc * r6p_2;
	r6p_2 = r6p_1;

	temp3 = r6pb * r6p_1;
	temp4 += temp3;

	temp3 = r6pa * sourc;
	r6p_1 = temp4 + temp3;
}

/* Fifth Formant of Parallel Vocal Tract */

static void reson5p _((void));

static void
reson5p()
{

	long temp3, temp4;

	temp4 = r5pc * r5p_2;
	r5p_2 = r5p_1;

	temp3 = r5pb * r5p_1;
	temp4 += temp3;

	temp3 = r5pa * sourc;
	r5p_1 = temp4 + temp3;
}

/* Fourth Formant of Parallel Vocal Tract */

static void reson4p _((void));

static void
reson4p()
{

	long temp3, temp4;

	temp4 = r4pc * r4p_2;
	r4p_2 = r4p_1;

	temp3 = r4pb * r4p_1;
	temp4 += temp3;

	temp3 = r4pa * sourc;
	r4p_1 = temp4 + temp3;
}

/* Third Formant of Parallel Vocal Tract */

static void reson3p _((void));

static void
reson3p()
{

	long temp3, temp4;

	temp4 = r3pc * r3p_2;
	r3p_2 = r3p_1;

	temp3 = r3pb * r3p_1;
	temp4 += temp3;

	temp3 = r3pa * sourc;
	r3p_1 = temp4 + temp3;
}

/* Second Formant of Parallel Vocal Tract */

static void reson2p _((void));

static void
reson2p()
{

	long temp3, temp4;

	temp4 = r2pc * r2p_2;
	r2p_2 = r2p_1;

	temp3 = r2pb * r2p_1;
	temp4 += temp3;

	temp3 = r2pa * sourc;
	r2p_1 = temp4 + temp3;
}


/* First Formant of Parallel Vocal Tract */

static void reson1p _((void));

static void
reson1p()
{

	long temp3, temp4;

	temp4 = r1pc * r1p_2;
	r1p_2 = r1p_1;

	temp3 = r1pb * r1p_1;
	temp4 += temp3;

	temp3 = r1pa * sourc;
	r1p_1 = temp4 + temp3;
}

/* Nasal Formant of Parallel Vocal Tract */

static void resonnpp _((void));

static void
resonnpp()
{

	long temp3, temp4;

	temp4 = rnppc * rnpp_2;
	rnpp_2 = rnpp_1;

	temp3 = rnppb * rnpp_1;
	temp4 += temp3;

	temp3 = rnppa * sourc;
	rnpp_1 = temp4 + temp3;
}




/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*								     */
/*	       S U B R O U T I N E   N O - R A D - C H A R	   */
/*								     */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define ACOEF	   0.005
#define BCOEF	   (1.0 - ACOEF)   /* Slight decay to remove dc */

static void no_rad_char _((double));

static void
no_rad_char(in)
	double	in;
{
	static double lastin;

	out = (ACOEF * in) + (BCOEF * lastin);
	lastin = out;
	out = -100. * out;      /* Scale up to make visible */
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*								     */
/*		  S U B R O U T I N E   G E T M A X		  */
/*								     */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Find absolute maximum of arg1 & arg2, save in arg2 */

static void getmax _((long, long *));

static void
getmax(arg1, arg2)
	long	arg1;
	long	*arg2;
{
	if (arg1 < 0)
		arg1 = -arg1;
	if (arg1 > *arg2)
		*arg2 = arg1;
}



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*								     */
/*		S U B R O U T I N E   P R - P A R S		  */
/*								     */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static void pr_pars _((pars_ty *));

static void
pr_pars(pp)
	pars_ty	*pp;
{
	static int esd;

	if (!esd)
	{
		esd++;
		error("Speaker-defining Constants:");
		error("OUTs = %ld", spkrdef.OUTs);
		error("Srat = %ld", spkrdef.Srat);
		error("NSfr = %ld", spkrdef.NSfr);
		error("Flp  = %ld", spkrdef.Flp);
		error("BWlp = %ld", spkrdef.BWlp);
		error("NFca = %ld", spkrdef.NFca);
		error("VSsw = %ld", spkrdef.VSsw);
	}

	error("Par values for this frame:");
	error("F0 = %hd", pp->F0hz10);
	error("AV = %hd", pp->AVdb);
	error("F1 = %hd", pp->F1hz);
	error("BW1 = %hd", pp->B1hz);
	error("F2 = %hd", pp->F2hz);
	error("BW2 = %hd", pp->B2hz);
	error("F3 = %hd", pp->F3hz);
	error("BW3 = %hd", pp->B3hz);
	error("F4 = %hd", pp->F4hz);
	error("BW4 = %hd", pp->B4hz);
	error("F5 = %hd", pp->F5hz);
	error("BW5 = %hd", pp->B5hz);
	error("F6 = %hd", pp->F6hz);
	error("BW6 = %hd", pp->B6hz);
	error("Fnz = %hd", pp->FNZhz);
	error("BWnz = %hd", pp->BNZhz);
	error("Fnp = %hd", pp->FNPhz);
	error("BWnp = %hd", pp->BNPhz);
	error("Aasp = %hd", pp->AP);
	error("Nopn = %hd", pp->Kopen);
	error("Atur = %hd", pp->Aturb);
	error("tilt = %hd", pp->TLTdb);
	error("Afrc = %hd", pp->AF);
	error("skew = %hd", pp->Kskew);
	error("A1 = %hd", pp->A1);
	error("BWp1 = %hd", pp->B1phz);
	error("A2 = %hd", pp->A2);
	error("BWp2 = %hd", pp->B2phz);
	error("A3 = %hd", pp->A3);
	error("BWp3 = %hd", pp->B3phz);
	error("A4 = %hd", pp->A4);
	error("BWp4 = %hd", pp->B4phz);
	error("A5 = %hd", pp->A5);
	error("BWp5 = %hd", pp->B5phz);
	error("A6 = %hd", pp->A6);
	error("BWp6 = %hd", pp->B6phz);
	error("AN = %hd", pp->ANP);
	error("AB = %hd", pp->AB);
	error("AVpa = %hd", pp->AVpdb);
	error("G0 = %hd", pp->Gain0);
}


static double dBconvert _((long));

static double
dBconvert(arg)
	long	arg;
{
	double	x;
	double	db;

	x = arg / 32767.;
	x = log10(x);
	db = 20. * x;
	return (db);
}


static void overload_warning _((long));

static void
overload_warning(arg)
	long		arg;
{
	static int	warnsw;

	if (!warnsw)
	{
		warnsw++;
		error
		(
"warning: signal at output of synthesizer (+%3.1f dB) exceeds \
0 dB at synthesis parameter frame %d",
			dBconvert(arg),
			disptcum / nspfr
		);
#if 0
		pr_pars();
#endif
	}
}


/*
 * NAME
 *	truncate
 *
 * SYNOPSIS
 *	long truncate(long n);
 *
 * DESCRIPTION
 *	The truncate funtion is used to
 *	truncate arg to fit into 16-bit word
 */

static long truncate _((long));

static long
truncate(arg) 
	long	arg; 
{
	if (arg < -32768)
	{
		overload_warning(-arg);
		arg = -32768;
	}
	if (arg >  32767)
	{
		overload_warning(arg);
		arg =  32767;
	}
	return arg;
}


static void resoncOut _((void));

static void
resoncOut()
{
	long	temp3;
	long	temp4;

	temp4 = routc * rout_2;	  /*   (ccoef * old2)     */
	rout_2 = rout_1;

	temp3 = routb * rout_1;	  /* + (bcoef * old1)     */
	temp4 += temp3;

	temp3 = routa * casc_next_in;   /* + (acoef * input)    */
	rout_1 = temp4 + temp3;
}


/*
 * NAME
 *	parwav
 *
 * SYNOPSIS
 *	void parwav(pars_ty *pars, char *jwave);
 *
 * DESCRIPTION
 *	The parwav function is used to
 *	convert a frame of parameter data to a waveform chunk
 */

void
parwav(pars, jwave) 
	pars_ty		*pars;
	short		*jwave; 
{
	trace(("parwav()\n{\n"/*}*/));
	/*
	 * Initialize synthesizer and get specification for current speech
	 * frame from host microcomputer
	 */
	gethost(pars);

	/*
	 * for each output sample of current frame:
	 */
	for (ns = 0; ns < nspfr; ns++)
	{
		/*
		 * Get low-passed random number for
		 * aspiration and frication noise
		 *
		 * output variable is 'noise'
		 */
		gen_noise();

		/*
		 * Amplitude modulate noise (reduce noise amplitude during
		 * second half of glottal period) if voicing simultaneously
		 * present
		 */
		if (nper > nmod)
			noise *= 0.5;

		/*
		 * Compute frication noise
		 */
		frics = amp_frica * noise;

		/*
		 * Compute voicing waveform
		 *
		 * (run glottal source simulation at 4 times normal
		 * sample rate to minimize quantization noise in period
		 * of female voice)
		 */
		for (n4 = 0; n4 < 4; n4++)
		{
			/*
			 * Use impulsive glottal source
			 */
			if (glsource == IMPULSIVE)
				voice = impulsive_source();

			/*
			 * Or use a more-natural-shaped source waveform with
			 * excitation occurring both upon opening and upon
			 * closure, strongest at closure
			 */
			else
				voice = natural_source();

			/*
			 * Reset period when counter 'nper' reaches T0
			 */
			if (nper >= T0)
			{
				nper = 0;
				pitch_synch_par_reset();
			}

			/*
			 * Low-pass filter voicing waveform before downsampling
			 * from 4*samrate to samrate samples/sec.
			 * Resonator f=.09*samrate, bw=.06*samrate
			 *
			 * in=voice, out=voice
			 */
			resonlp();

			/*
			 * Increment counter that keeps track
			 * of 4*samrate samples/sec
			 */
			nper++;
		}

		/*
		 * Tilt spectrum of voicing source down by soft low-pass
		 * filtering, amount of tilt determined by TLTdb
		 */
		voice = (voice * onemd) + (vlast * decay);
		vlast = voice;

		/*
		 * Add breathiness during glottal open phase
		 */
		if (nper < nopen)
		{
			/*
			 * Amount of breathiness determined by parameter
			 * Aturb.  Use nrand rather than noise because
			 * noise is low-passed
			 */
			voice += amp_breth * nrand;
		}

		/*
		 * Set amplitude of voicing
		 */
		glotout = amp_voice * voice;
		par_glotout = par_amp_voice * voice;

		/*
		 * Compute aspiration amplitude and add to voicing source
		 */
		aspiration = amp_aspir * noise;
		glotout += aspiration;
		par_glotout += aspiration;

		/*
		 * Cascade vocal tract, excited by laryngeal sources.
		 * Nasal antiresonator, then
		 * formants FNP, F5, F4, F3, F2, F1
		 *
		 * in=glotout, out=rnzout
		 */
		resoncnz();

		/* in=rnzout, out=rnpc_1 */
		resoncnp();

		casc_next_in = rnpc_1;

		if (nfcascade >= 8)
		{
			/* Do not use unless samrat = 16000 */
			resonc8();
			casc_next_in = r8c_1;
		}

		if (nfcascade >= 7)
		{
			/* Do not use unless samrat = 16000 */
			resonc7();
			casc_next_in = r7c_1;
		}

		if (nfcascade >= 6)
		{
			/* Do not use unless long vocal tract or */
			resonc6();
			/* samrat increased */
			casc_next_in = r6c_1;
		}

		if (nfcascade >= 5)
		{
			resonc5();
			casc_next_in = r5c_1;
		}

		if (nfcascade >= 4)
		{
			resonc4();
			casc_next_in = r4c_1;
		}

		if (nfcascade >= 3)
		{
			resonc3();
			casc_next_in = r3c_1;
		}

		if (nfcascade >= 2)
		{
			resonc2();
			casc_next_in = r2c_1;
		}

		if (nfcascade >= 1)
		{
			resonc1();
		}

		if (outsl > 0)
		{
			if (outsl < 13)
			{
				if (outsl ==  1)
					out = voice;
				if (outsl ==  2)
					out = aspiration;
				if (outsl ==  3)
					out = frics;
				if (outsl ==  4)
					out = glotout * 2.0;
				if (outsl ==  5)
					out = par_glotout;
				if (outsl ==  6)
					out = rnzout;
				if (outsl ==  7)
					out = rnpc_1;
				if (outsl ==  8)
					out = r5c_1;
				if (outsl ==  9)
					out = r4c_1;
				if (outsl == 10)
					out = r3c_1;
				if (outsl == 11)
					out = r2c_1;
				if (outsl == 12)
					out = r1c_1;
				if (outsl <= 4)
				{
					/*
					* Take out effects of
					* radiation char
					*/
					no_rad_char(out);
				}
				goto skip;
			}
		}
		out = r1c_1;

		/*
		 * Excite parallel F1 and FNP by voicing waveform
		 */
		/* Source is voicing plus aspiration */
		sourc = par_glotout;
		/* in=sourc, out=rnpp_1 */
		reson1p();
		/* in=sourc, out=r1c_1 */
		resonnpp();
		/* Add in phase, boost lows for nasalized */
		out += rnpp_1 + r1p_1;

		trace(("count = %d, out = %f\n", count, out));

		/*
		 * Standard parallel vocal tract
		 * Formants F6, F5, F4, F3, F2,
		 * outputs added with alternating sign
		 *
		 * Sound sourc for other parallel resonators is
		 * frication plus first difference of voicing waveform
		 */
		sourc = frics + par_glotout - glotlast;
		glotlast = par_glotout;

		trace(("count = %d sourc = %f\n", count, sourc));
/* ??? */	count++;
 
		reson6p();	      /* in=sourc, out=r6p_1 */
		out = r6p_1 - out;

		reson5p();	      /* in=sourc, out=r5p_1 */
		out = r5p_1 - out;

		reson4p();	      /* in=sourc, out=r4p_1 */
		out = r4p_1 - out;

		reson3p();	      /* in=sourc, out=r3p_1 */
		out = r3p_1 - out;

		reson2p();	      /* in=sourc, out=r2p_1 */
		out = r2p_1 - out;

		outbypas = amp_bypas * sourc;
		out = outbypas - out;

		if (outsl > 12)
		{
			if (outsl == 13)
				out = r6p_1;
			if (outsl == 14)
				out = r5p_1;
			if (outsl == 15)
				out = r4p_1;
			if (outsl == 16)
				out = r3p_1;
			if (outsl == 17)
				out = r2p_1;
			if (outsl == 18)
				out = r1p_1;
			if (outsl == 19)
				out = rnpp_1;
			if (outsl == 20)
				out = outbypas;
		}

		casc_next_in = out;
		resoncOut();
		out = rout_1;

skip:
		temp = out * amp_gain0;	 /* Convert back to integer */
		getmax(temp, &sigmx);

#if 0
		if (abs(temp) > 4500)
			temp = 4500;
#endif
		/* Truncate if exceeds 16 bits */
		*jwave++ = truncate(temp);
	}
trace((/*{*/"}\n"));
}
