/****************************************************************************
** libebml : parse EBML files, see http://embl.sourceforge.net/
**
** <file/class description>
**
** Copyright (C) 2002-2003 Steve Lhomme.  All rights reserved.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding an other license may use this file in accordance with 
** the Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.matroska.org/license/qpl/ for QPL licensing information.
** See http://www.matroska.org/license/gpl/ for GPL licensing information.
**
** Contact license@matroska.org if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

/*!
	\file
	\version \$Id: WinIOCallback.cpp,v 1.10 2003/10/08 15:47:58 jcsston Exp $
	\author Steve Lhomme     <robux4 @ users.sf.net>
	\author Jory Stone       <jcsston @ toughguy.net>
	\author Cyrius           <suiryc @ users.sf.net>
*/

#include <cassert>

#include "WinIOCallback.h"

#include "ebml/Debug.h"

#ifndef INVALID_SET_FILE_POINTER // found in newer platform SDKs
#define INVALID_SET_FILE_POINTER ((DWORD)-1)
#endif // INVALID_SET_FILE_POINTER

START_LIBEBML_NAMESPACE

WinIOCallback::WinIOCallback(const char* Path, const open_mode aMode, DWORD dwFlags)
	:mFile(NULL), mOk(false)
{
	mOk = open(Path, aMode, dwFlags);	
}

WinIOCallback::WinIOCallback(const wchar_t* Path, const open_mode aMode, DWORD dwFlags)
	:mFile(NULL), mOk(false)
{
	mOk = open(Path, aMode, dwFlags);
}

WinIOCallback::~WinIOCallback()
{
	close();
}

bool WinIOCallback::open(const char* Path, const open_mode aMode, DWORD dwFlags) 
{
	assert(Path!=0);

	DWORD AccessMode, ShareMode, Disposition;

	switch (aMode)
	{
	case MODE_READ:
		AccessMode = GENERIC_READ;
		ShareMode = FILE_SHARE_READ|FILE_SHARE_WRITE;
		Disposition = OPEN_EXISTING;
		break;
	case MODE_WRITE:
		AccessMode = GENERIC_WRITE;
		ShareMode = 0;
		Disposition = OPEN_ALWAYS;
		break;
	case MODE_SAFE:
		AccessMode = GENERIC_WRITE|GENERIC_READ;
		ShareMode = FILE_SHARE_READ|FILE_SHARE_WRITE;
		Disposition = OPEN_ALWAYS;
		break;
	case MODE_CREATE:
		AccessMode = GENERIC_WRITE;
		ShareMode = 0;
		Disposition = CREATE_ALWAYS;
		break;
	default:
		assert(false);
	}

	mFile = CreateFileA(Path, AccessMode, ShareMode, NULL, Disposition, dwFlags, NULL);
	if ((mFile == INVALID_HANDLE_VALUE) || ((long)mFile == 0xffffffff))
	{
		//File was not opened
		char err_msg[256];
		DWORD error_code = GetLastError();
		// An error message about the file already existing is not really an error message :P
		if (error_code != ERROR_ALREADY_EXISTS) {
			FormatMessageA(FORMAT_MESSAGE_FROM_SYSTEM, 0, error_code, 0, err_msg, 255, NULL);
			EBML_TRACE("Failed to open file \"%hs\" in mode %d.", Path, aMode);						

			mLastErrorStr = err_msg;
			return mOk = false;
		}
	}

	EBML_TRACE("Successfully opened file \"%hs\" in mode %d. The handle is %p\n", Path, aMode, mFile);

	return mOk = true;
};

bool WinIOCallback::open(const wchar_t* Path, const open_mode aMode, DWORD dwFlags) 
{
	assert(Path!=0);

	DWORD AccessMode, ShareMode, Disposition;

	switch (aMode)
	{
	case MODE_READ:
		AccessMode = GENERIC_READ;
		ShareMode = FILE_SHARE_READ|FILE_SHARE_WRITE;
		Disposition = OPEN_EXISTING;
		break;
	case MODE_WRITE:
		AccessMode = GENERIC_WRITE;
		ShareMode = 0;
		Disposition = OPEN_ALWAYS;
		break;
	case MODE_SAFE:
		AccessMode = GENERIC_WRITE|GENERIC_READ;
		ShareMode = FILE_SHARE_READ|FILE_SHARE_WRITE;
		Disposition = OPEN_ALWAYS;
		break;
	case MODE_CREATE:
		AccessMode = GENERIC_WRITE;
		ShareMode = 0;
		Disposition = CREATE_ALWAYS;
		break;
	default:
		assert(false);
	}

	if ((LONG)GetVersion() >= 0) {
		mFile = CreateFileW(Path, AccessMode, ShareMode, NULL, Disposition, dwFlags, NULL);
	} else {
		int errCode;
		unsigned int bufferSize = wcslen(Path) + sizeof(wchar_t) * 2;
		std::string PathA;
		PathA.resize(bufferSize);
		errCode = WideCharToMultiByte(CP_ACP, 0, Path, wcslen(Path), (char *)PathA.c_str(), bufferSize, NULL, NULL);
		if (errCode == 0)
			errCode = GetLastError();
#ifdef _DEBUG		
		if (errCode == ERROR_INSUFFICIENT_BUFFER) OutputDebugString(TEXT("WinIOCallback::WideCharToMultiByte::ERROR_INSUFFICIENT_BUFFER"));
		if (errCode == ERROR_INVALID_FLAGS) OutputDebugString(TEXT("WinIOCallback::WideCharToMultiByte::ERROR_INVALID_FLAGS"));
		if (errCode == ERROR_INVALID_PARAMETER) OutputDebugString(TEXT("WinIOCallback::WideCharToMultiByte::ERROR_INVALID_PARAMETER"));
#endif
		while (errCode == ERROR_INSUFFICIENT_BUFFER) {
			// Increase the buffer size
			bufferSize += MAX_PATH;
			PathA.resize(bufferSize);
			errCode = WideCharToMultiByte(CP_ACP, WC_SEPCHARS, Path, wcslen(Path), (char *)PathA.c_str(), bufferSize, NULL, NULL);
			if (errCode == 0)
				errCode = GetLastError();
		}
		if (errCode != 0) {
			mFile = CreateFileA(PathA.c_str(), AccessMode, ShareMode, NULL, Disposition, dwFlags, NULL);
		} else {
			mLastErrorStr = "Couldn't convert Unicode filename to ANSI.";			
			return mOk = false;
		}
	}
	if ((mFile == INVALID_HANDLE_VALUE) || ((long)mFile == 0xffffffff))
	{
		//File was not opened
		char err_msg[256];
		DWORD error_code = GetLastError();
		// An error message about the file already existing is not really an error message :P
		if (error_code != ERROR_ALREADY_EXISTS) {
			FormatMessageA(FORMAT_MESSAGE_FROM_SYSTEM, 0, error_code, 0, err_msg, 255, NULL);
			EBML_TRACE("Failed to open file \"%S\" in mode %d.", Path, aMode);			
			mLastErrorStr = err_msg;
			return mOk = false;
		}		
	}

	EBML_TRACE("Successfully opened file \"%S\" in mode %d. The handle is %p\n", Path, aMode, mFile);
	return mOk = true;
}

void WinIOCallback::close()
{
	if (mFile) {
		CloseHandle(mFile);
		mFile = NULL;
	}
}

uint64 WinIOCallback::getFilePointer()
{
	if (!mFile) {
		return 0;
	}
	LONG High = 0;
	DWORD Low = SetFilePointer(mFile, 0, &High, FILE_CURRENT);
	if ( (Low==INVALID_SET_FILE_POINTER) && (GetLastError()!=NO_ERROR) )
		return -1;
	return ((uint64(High)<<32) | Low);
}

void WinIOCallback::setFilePointer(int64 Offset, seek_mode Mode)
{
	DWORD Method;
	switch(Mode)
	{
	case seek_beginning:
		Method=FILE_BEGIN;
		break;
	case seek_current:
		Method=FILE_CURRENT;
		break;
	case seek_end:
		Method=FILE_END;
		break;
	default:
		assert(false);
		break;
	}

	LONG High = LONG(Offset>>32);
	SetFilePointer(mFile, LONG(Offset & 0xffffffff), &High, Method);
}

uint32 WinIOCallback::read(void*Buffer,size_t Size)
{
	DWORD BytesRead;
	if (!ReadFile(mFile, Buffer, Size, &BytesRead, NULL)) {
		return 0;
	}
	return BytesRead;
}

size_t WinIOCallback::write(const void*Buffer,size_t Size)
{
	DWORD BytesWriten;
	if (!WriteFile(mFile, Buffer, Size, &BytesWriten, NULL)) {
		return 0;
	}
	return BytesWriten;
}

END_LIBEBML_NAMESPACE
