/*****************************************************************************
 * open.cpp : wxWindows plugin for vlc
 *****************************************************************************
 * Copyright (C) 2000-2004 VideoLAN
 * $Id: open.cpp 7728 2004-05-20 10:16:52Z gbazin $
 *
 * Authors: Gildas Bazin <gbazin@videolan.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

/*****************************************************************************
 * Preamble
 *****************************************************************************/
#include <stdlib.h>                                      /* malloc(), free() */
#include <errno.h>                                                 /* ENOMEM */
#include <string.h>                                            /* strerror() */
#include <stdio.h>

#include <vlc/vlc.h>

#include <wx/combobox.h>
#include <wx/statline.h>
#include <wx/tokenzr.h>

#include <vlc/intf.h>

#include "wxwindows.h"
#include "preferences_widgets.h"

#ifndef wxRB_SINGLE
#   define wxRB_SINGLE 0
#endif

/*****************************************************************************
 * Event Table.
 *****************************************************************************/

/* IDs for the controls and the menu commands */
enum
{
    Notebook_Event = wxID_HIGHEST,
    MRL_Event,

    FileBrowse_Event,
    FileName_Event,

    DiscType_Event,
    DiscDevice_Event,
    DiscTitle_Event,
    DiscChapter_Event,

    NetType_Event,
    NetRadio1_Event, NetRadio2_Event, NetRadio3_Event, NetRadio4_Event,
    NetPort1_Event, NetPort2_Event, NetPort3_Event,
    NetAddr1_Event, NetAddr2_Event, NetAddr3_Event, NetAddr4_Event,
    NetForceIPv6_Event,

    SubsFileEnable_Event,
    SubsFileSettings_Event,

    SoutEnable_Event,
    SoutSettings_Event,

    AdvancedOptions_Event
};

BEGIN_EVENT_TABLE(OpenDialog, wxDialog)
    /* Button events */
    EVT_BUTTON(wxID_OK, OpenDialog::OnOk)
    EVT_BUTTON(wxID_CANCEL, OpenDialog::OnCancel)

    EVT_NOTEBOOK_PAGE_CHANGED(Notebook_Event, OpenDialog::OnPageChange)

    EVT_TEXT(MRL_Event, OpenDialog::OnMRLChange)

    /* Events generated by the file panel */
    EVT_TEXT(FileName_Event, OpenDialog::OnFilePanelChange)
    EVT_BUTTON(FileBrowse_Event, OpenDialog::OnFileBrowse)

    /* Events generated by the disc panel */
    EVT_RADIOBOX(DiscType_Event, OpenDialog::OnDiscTypeChange)
    EVT_TEXT(DiscDevice_Event, OpenDialog::OnDiscDeviceChange)
    EVT_TEXT(DiscDevice_Event, OpenDialog::OnDiscPanelChange)
    EVT_TEXT(DiscTitle_Event, OpenDialog::OnDiscPanelChange)
    EVT_SPINCTRL(DiscTitle_Event, OpenDialog::OnDiscPanelChange)
    EVT_TEXT(DiscChapter_Event, OpenDialog::OnDiscPanelChange)
    EVT_SPINCTRL(DiscChapter_Event, OpenDialog::OnDiscPanelChange)

    /* Events generated by the net panel */
    EVT_RADIOBUTTON(NetRadio1_Event, OpenDialog::OnNetTypeChange)
    EVT_RADIOBUTTON(NetRadio2_Event, OpenDialog::OnNetTypeChange)
    EVT_RADIOBUTTON(NetRadio3_Event, OpenDialog::OnNetTypeChange)
    EVT_RADIOBUTTON(NetRadio4_Event, OpenDialog::OnNetTypeChange)
    EVT_TEXT(NetPort1_Event, OpenDialog::OnNetPanelChange)
    EVT_SPINCTRL(NetPort1_Event, OpenDialog::OnNetPanelChange)
    EVT_TEXT(NetPort2_Event, OpenDialog::OnNetPanelChange)
    EVT_SPINCTRL(NetPort2_Event, OpenDialog::OnNetPanelChange)
    EVT_TEXT(NetPort3_Event, OpenDialog::OnNetPanelChange)
    EVT_SPINCTRL(NetPort3_Event, OpenDialog::OnNetPanelChange)
    EVT_TEXT(NetAddr2_Event, OpenDialog::OnNetPanelChange)
    EVT_TEXT(NetAddr3_Event, OpenDialog::OnNetPanelChange)
    EVT_TEXT(NetAddr4_Event, OpenDialog::OnNetPanelChange)
    EVT_CHECKBOX(NetForceIPv6_Event, OpenDialog::OnNetPanelChange)

    /* Events generated by the subtitle file buttons */
    EVT_CHECKBOX(SubsFileEnable_Event, OpenDialog::OnSubsFileEnable)
    EVT_BUTTON(SubsFileSettings_Event, OpenDialog::OnSubsFileSettings)

    /* Events generated by the stream output buttons */
    EVT_CHECKBOX(SoutEnable_Event, OpenDialog::OnSoutEnable)
    EVT_BUTTON(SoutSettings_Event, OpenDialog::OnSoutSettings)

    /* Hide the window when the user closes the window */
    EVT_CLOSE(OpenDialog::OnCancel)

END_EVENT_TABLE()

/*****************************************************************************
 * AutoBuiltPanel.
 *****************************************************************************/
WX_DEFINE_ARRAY(ConfigControl *, ArrayOfConfigControls);

class AutoBuiltPanel : public wxPanel
{
public:

    AutoBuiltPanel() { }
    AutoBuiltPanel( wxWindow *, OpenDialog *, intf_thread_t *,
                    const module_t * );

    virtual ~AutoBuiltPanel() {}

    void UpdateAdvancedMRL();

    wxString name;
    ArrayOfConfigControls config_array;
    ArrayOfConfigControls advanced_config_array;
    wxComboBox *p_advanced_mrl_combo;

private:
    intf_thread_t *p_intf;
    OpenDialog *p_open_dialog;

    void OnAdvanced( wxCommandEvent& event );
    wxDialog *p_advanced_dialog;

    DECLARE_EVENT_TABLE();
};

BEGIN_EVENT_TABLE(AutoBuiltPanel, wxPanel)
    EVT_BUTTON(AdvancedOptions_Event, AutoBuiltPanel::OnAdvanced)
END_EVENT_TABLE()

static void AutoBuildCallback( void *p_data )
{
    ((OpenDialog *)p_data)->UpdateMRL();
}

static void AutoBuildAdvancedCallback( void *p_data )
{
    ((AutoBuiltPanel *)p_data)->UpdateAdvancedMRL();
}

AutoBuiltPanel::AutoBuiltPanel( wxWindow *parent, OpenDialog *dialog,
                                intf_thread_t *_p_intf,
                                const module_t *p_module )
  : wxPanel( parent, -1, wxDefaultPosition, wxDefaultSize ),
    name( wxU(p_module->psz_object_name) ),
    p_advanced_mrl_combo( NULL ),
    p_intf( _p_intf ), p_open_dialog( dialog ), p_advanced_dialog( NULL )
{
    wxBoxSizer *sizer = new wxBoxSizer( wxVERTICAL );
    module_config_t *p_item = p_module->p_config;
    bool b_advanced = false;

    if( p_item ) do
    {
        if( !(p_item->i_type & CONFIG_HINT) && p_item->b_advanced )
            b_advanced = true;

        if( p_item->i_type & CONFIG_HINT || p_item->b_advanced )
            continue;

        ConfigControl *control =
            CreateConfigControl( VLC_OBJECT(p_intf), p_item, this );

        config_array.Add( control );

        /* Don't add items that were not recognized */
        if( control == NULL ) continue;

        control->SetUpdateCallback( AutoBuildCallback, (void *)dialog );

        sizer->Add( control, 0, wxEXPAND | wxALL, 2 );
    }
    while( p_item->i_type != CONFIG_HINT_END && p_item++ );

    if( b_advanced )
    {
        wxPanel *dummy_panel = new wxPanel( this, -1 );
        sizer->Add( dummy_panel, 1 );

        wxButton *button =
            new wxButton( this, AdvancedOptions_Event,
                          wxU(_("Advanced options...")) );
        sizer->Add( button, 0, wxALL, 5 );

        /* Build the advanced dialog */
        p_advanced_dialog =
            new wxDialog( this, -1, ((wxString)wxU(_("Advanced options"))) +
                          wxT(" (") + wxU( p_module->psz_longname ) + wxT(")"),
                          wxDefaultPosition, wxDefaultSize,
                          wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER );

        wxBoxSizer *sizer = new wxBoxSizer( wxVERTICAL );

        /* Create MRL combobox */
        wxBoxSizer *mrl_sizer_sizer = new wxBoxSizer( wxHORIZONTAL );
        wxStaticBox *mrl_box =
            new wxStaticBox( p_advanced_dialog, -1,
                             wxU(_("Advanced options")) );
        wxStaticBoxSizer *mrl_sizer =
            new wxStaticBoxSizer( mrl_box, wxHORIZONTAL );
        wxStaticText *mrl_label =
            new wxStaticText( p_advanced_dialog, -1, wxU(_("Options:")) );
        p_advanced_mrl_combo =
            new wxComboBox( p_advanced_dialog, MRL_Event, wxT(""),
                            wxDefaultPosition, wxDefaultSize );
        mrl_sizer->Add( mrl_label, 0, wxALL | wxALIGN_CENTER, 5 );
        mrl_sizer->Add( p_advanced_mrl_combo, 1, wxALL | wxALIGN_CENTER, 5 );
        mrl_sizer_sizer->Add( mrl_sizer, 1, wxEXPAND | wxALL, 5 );
        sizer->Add( mrl_sizer_sizer, 0, wxEXPAND | wxALL, 2 );

        /* Add advanced options to panel */
        module_config_t *p_item = p_module->p_config;
        if( p_item ) do
        {
            if( p_item->i_type & CONFIG_HINT || !p_item->b_advanced )
                continue;

            ConfigControl *control =
                CreateConfigControl( VLC_OBJECT(p_intf), p_item,
                                     p_advanced_dialog );

            advanced_config_array.Add( control );

            /* Don't add items that were not recognized */
            if( control == NULL ) continue;

            control->SetUpdateCallback( AutoBuildAdvancedCallback,
                                        (void *)this );

            sizer->Add( control, 0, wxEXPAND | wxALL, 2 );
        }
        while( p_item->i_type != CONFIG_HINT_END && p_item++ );

        /* Separation */
        dummy_panel = new wxPanel( p_advanced_dialog, -1 );
        sizer->Add( dummy_panel, 1 );
        wxStaticLine *static_line =
            new wxStaticLine( p_advanced_dialog, wxID_OK );
        sizer->Add( static_line, 0, wxEXPAND | wxALL, 5 );

        /* Create buttons */
        wxButton *ok_button =
            new wxButton( p_advanced_dialog, wxID_OK, wxU(_("OK")) );
        ok_button->SetDefault();
        wxButton *cancel_button =
            new wxButton( p_advanced_dialog, wxID_CANCEL, wxU(_("Cancel")) );
        wxBoxSizer *button_sizer = new wxBoxSizer( wxHORIZONTAL );
        button_sizer->Add( ok_button, 0, wxALL, 5 );
        button_sizer->Add( cancel_button, 0, wxALL, 5 );
        button_sizer->Layout();
        sizer->Add( button_sizer, 0, wxALL, 0 );

        sizer->SetMinSize( 400, -1 );
        p_advanced_dialog->SetSizerAndFit( sizer );
    }

    this->SetSizerAndFit( sizer );
}

void AutoBuiltPanel::OnAdvanced( wxCommandEvent& event )
{
    if( p_advanced_dialog->ShowModal() == wxID_OK )
    {
        UpdateAdvancedMRL();
        p_open_dialog->UpdateMRL();
    }
}

void AutoBuiltPanel::UpdateAdvancedMRL()
{
    wxString mrltemp;

    for( int i = 0; i < (int)advanced_config_array.GetCount(); i++ )
    {
        ConfigControl *control = advanced_config_array.Item(i);

        mrltemp += (i ? wxT(" :") : wxT(":"));

        if( control->GetType() == CONFIG_ITEM_BOOL &&
            !control->GetIntValue() ) mrltemp += wxT("no-");

        mrltemp += control->GetName();

        switch( control->GetType() )
        {
        case CONFIG_ITEM_STRING:
        case CONFIG_ITEM_FILE:
        case CONFIG_ITEM_DIRECTORY:
        case CONFIG_ITEM_MODULE:
            mrltemp += wxT("=\"") + control->GetPszValue() + wxT("\"");
            break;
        case CONFIG_ITEM_INTEGER:
            mrltemp +=
                wxString::Format( wxT("=%i"), control->GetIntValue() );
            break;
        case CONFIG_ITEM_FLOAT:
            mrltemp +=
                wxString::Format(wxT("=%f"), control->GetFloatValue());
            break;
        }
    }

    p_advanced_mrl_combo->SetValue( mrltemp );
}

/*****************************************************************************
 * Constructor.
 *****************************************************************************/
OpenDialog::OpenDialog( intf_thread_t *_p_intf, wxWindow *_p_parent,
                        int i_access_method, int i_arg ):
      wxDialog( _p_parent, -1, wxU(_("Open...")), wxDefaultPosition,
             wxDefaultSize, wxDEFAULT_FRAME_STYLE )
{
    OpenDialog( _p_intf, _p_parent, i_access_method, i_arg, OPEN_NORMAL );
}

OpenDialog::OpenDialog( intf_thread_t *_p_intf, wxWindow *_p_parent,
                        int i_access_method, int i_arg, int _i_method ):
      wxDialog( _p_parent, -1, wxU(_("Open...")), wxDefaultPosition,
             wxDefaultSize, wxDEFAULT_FRAME_STYLE )
{
    /* Initializations */
    i_method = _i_method;
    p_intf = _p_intf;
    p_parent = _p_parent;
    SetIcon( *p_intf->p_sys->p_icon );
    file_dialog = NULL;
    i_disc_type_selection = 0;
    i_open_arg = i_arg;

    sout_dialog = NULL;
    subsfile_dialog = NULL;
    b_disc_device_changed = false;

    /* Create a panel to put everything in */
    wxPanel *panel = new wxPanel( this, -1 );
    panel->SetAutoLayout( TRUE );

    /* Create MRL combobox */
    wxBoxSizer *mrl_sizer_sizer = new wxBoxSizer( wxHORIZONTAL );
    wxStaticBox *mrl_box = new wxStaticBox( panel, -1,
                               wxU(_("Media Resource Locator (MRL)")) );
    wxStaticBoxSizer *mrl_sizer = new wxStaticBoxSizer( mrl_box,
                                                        wxHORIZONTAL );
    wxStaticText *mrl_label = new wxStaticText( panel, -1,
                                                wxU(_("Open:")) );
    mrl_combo = new wxComboBox( panel, MRL_Event, wxT(""),
                                wxPoint(20,25), wxSize(120, -1),
                                0, NULL );
    mrl_combo->SetToolTip( wxU(_("You can use this field directly by typing "
        "the full MRL you want to open.\n""Alternatively, the field will be "
        "filled automatically when you use the controls below.")) );

    mrl_sizer->Add( mrl_label, 0, wxALL | wxALIGN_CENTER, 5 );
    mrl_sizer->Add( mrl_combo, 1, wxALL | wxALIGN_CENTER, 5 );
    mrl_sizer_sizer->Add( mrl_sizer, 1, wxEXPAND | wxALL, 5 );


    /* Create Static Text */
    wxStaticText *label = new wxStaticText( panel, -1,
        wxU(_("Alternatively, you can build an MRL using one of the "
              "following predefined targets:")) );

    wxFlexGridSizer *sout_sizer = NULL;
    wxStaticLine *static_line = NULL;

    if( i_method == OPEN_NORMAL )
    {
        /* Create Stream Output checkox */
        sout_sizer = new wxFlexGridSizer( 2, 1, 20 );


        sout_checkbox = new wxCheckBox( panel, SoutEnable_Event,
                                         wxU(_("Stream output")) );
        sout_checkbox->SetToolTip( wxU(_("Use VLC as a server of streams")) );
        sout_sizer->Add( sout_checkbox, 0,
                         wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
        sout_button = new wxButton( panel, SoutSettings_Event,
                                    wxU(_("Settings...")) );
        sout_button->Disable();

        char *psz_sout = config_GetPsz( p_intf, "sout" );
        if( psz_sout && *psz_sout )
        {
            sout_checkbox->SetValue(TRUE);
            sout_button->Enable();
            subsfile_mrl.Add( wxString(wxT("sout=")) + wxL2U(psz_sout) );
        }
        if( psz_sout ) free( psz_sout );

        sout_sizer->Add( sout_button, 1, wxALIGN_LEFT |
                         wxALIGN_CENTER_VERTICAL );

        /* Separation */
        static_line = new wxStaticLine( panel, wxID_OK );
    }

    /* Create the buttons */
    wxButton *ok_button = new wxButton( panel, wxID_OK, wxU(_("OK")) );
    ok_button->SetDefault();
    wxButton *cancel_button = new wxButton( panel, wxID_CANCEL,
                                            wxU(_("Cancel")) );

    /* Create notebook */
    notebook = new wxNotebook( panel, Notebook_Event );
    wxNotebookSizer *notebook_sizer = new wxNotebookSizer( notebook );

    notebook->AddPage( FilePanel( notebook ), wxU(_("File")),
                       i_access_method == FILE_ACCESS );
    notebook->AddPage( DiscPanel( notebook ), wxU(_("Disc")),
                       i_access_method == DISC_ACCESS );
    notebook->AddPage( NetPanel( notebook ), wxU(_("Network")),
                       i_access_method == NET_ACCESS );

    module_t *p_module = config_FindModule( VLC_OBJECT(p_intf), "v4l" );
    if( p_module )
    {
        AutoBuiltPanel *autopanel =
            new AutoBuiltPanel( notebook, this, p_intf, p_module );
        input_tab_array.Add( autopanel );
        notebook->AddPage( autopanel, wxU( p_module->psz_shortname ),
                           i_access_method == CAPTURE_ACCESS );
    }

    p_module = config_FindModule( VLC_OBJECT(p_intf), "pvr" );
    if( p_module )
    {
        AutoBuiltPanel *autopanel =
            new AutoBuiltPanel( notebook, this, p_intf, p_module );
        input_tab_array.Add( autopanel );
        notebook->AddPage( autopanel, wxU( p_module->psz_shortname ),
                           i_access_method == CAPTURE_ACCESS );
    }

    p_module = config_FindModule( VLC_OBJECT(p_intf), "dvb" );
    if( p_module )
    {
        AutoBuiltPanel *autopanel =
            new AutoBuiltPanel( notebook, this, p_intf, p_module );
        input_tab_array.Add( autopanel );
        notebook->AddPage( autopanel, wxU( p_module->psz_shortname ),
                           i_access_method == CAPTURE_ACCESS );
    }

    p_module = config_FindModule( VLC_OBJECT(p_intf), "dshow" );
    if( p_module )
    {
        AutoBuiltPanel *autopanel =
            new AutoBuiltPanel( notebook, this, p_intf, p_module );
        input_tab_array.Add( autopanel );
        notebook->AddPage( autopanel, wxU( p_module->psz_shortname ),
                           i_access_method == CAPTURE_ACCESS );
    }

    /* Update Disc panel */
    wxCommandEvent dummy_event;
    OnDiscTypeChange( dummy_event );

    /* Update Net panel */
    dummy_event.SetId( NetRadio1_Event );
    OnNetTypeChange( dummy_event );

    /* Update MRL */
    wxNotebookEvent event( wxEVT_NULL, 0, i_access_method );
    OnPageChange( event );

    /* Place everything in sizers */
    wxBoxSizer *button_sizer = new wxBoxSizer( wxHORIZONTAL );
    button_sizer->Add( ok_button, 0, wxALL, 5 );
    button_sizer->Add( cancel_button, 0, wxALL, 5 );
    button_sizer->Layout();
    wxBoxSizer *main_sizer = new wxBoxSizer( wxVERTICAL );
    wxBoxSizer *panel_sizer = new wxBoxSizer( wxVERTICAL );
    panel_sizer->Add( mrl_sizer_sizer, 0, wxEXPAND, 5 );
    panel_sizer->Add( label, 0, wxEXPAND | wxALL, 5 );
    panel_sizer->Add( notebook_sizer, 1, wxEXPAND | wxALL, 5 );
    if( i_method == OPEN_NORMAL)
    {
        panel_sizer->Add( sout_sizer, 0, wxALIGN_LEFT | wxALL, 5 );
        panel_sizer->Add( static_line, 0, wxEXPAND | wxALL, 5 );
    }
    panel_sizer->Add( button_sizer, 0, wxALIGN_LEFT | wxALL, 5 );
    panel_sizer->Layout();
    panel->SetSizerAndFit( panel_sizer );
    main_sizer->Add( panel, 1, wxGROW, 0 );
    main_sizer->Layout();
    SetSizerAndFit( main_sizer );
}

OpenDialog::~OpenDialog()
{
    /* Clean up */
    if( file_dialog ) delete file_dialog;
    if( sout_dialog ) delete sout_dialog;
    if( subsfile_dialog ) delete subsfile_dialog;
}

int OpenDialog::Show( int i_access_method, int i_arg )
{
    notebook->SetSelection( i_access_method );
    int i_ret = wxDialog::Show();
    Raise();
    SetFocus();
    i_open_arg = i_arg;
    return i_ret;
}

int OpenDialog::Show()
{
    int i_ret = wxDialog::Show();
    Raise();
    SetFocus();
    return i_ret;
}

/*****************************************************************************
 * Private methods.
 *****************************************************************************/
wxPanel *OpenDialog::FilePanel( wxWindow* parent )
{
    wxPanel *panel = new wxPanel( parent, -1, wxDefaultPosition,
                                  wxSize(200, 200) );

    wxBoxSizer *sizer = new wxBoxSizer( wxVERTICAL );

    /* Create browse file line */
    wxBoxSizer *file_sizer = new wxBoxSizer( wxHORIZONTAL );

    file_combo = new wxComboBox( panel, FileName_Event, wxT(""),
                                 wxPoint(20,25), wxSize(200, -1), 0, NULL );
    wxButton *browse_button = new wxButton( panel, FileBrowse_Event,
                                            wxU(_("Browse...")) );
    file_sizer->Add( file_combo, 1, wxALL, 5 );
    file_sizer->Add( browse_button, 0, wxALL, 5 );

    /* Create Subtitles File checkox */
    wxFlexGridSizer *subsfile_sizer = new wxFlexGridSizer( 2, 1, 20 );
    subsfile_checkbox = new wxCheckBox( panel, SubsFileEnable_Event,
                                        wxU(_("Subtitle options")) );
    subsfile_checkbox->SetToolTip( wxU(_("Force options for separate subtitle files.")) );
    subsfile_sizer->Add( subsfile_checkbox, 0,
                         wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
    subsfile_button = new wxButton( panel, SubsFileSettings_Event,
                                    wxU(_("Settings...")) );
    subsfile_button->Disable();

    char *psz_subsfile = config_GetPsz( p_intf, "sub-file" );
    if( psz_subsfile && *psz_subsfile )
    {
        subsfile_checkbox->SetValue(TRUE);
        subsfile_button->Enable();
        subsfile_mrl.Add( wxString(wxT("sub-file=")) + wxL2U(psz_subsfile) );
    }
    if( psz_subsfile ) free( psz_subsfile );

    subsfile_sizer->Add( subsfile_button, 1,
                         wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );

    sizer->Add( file_sizer, 0, wxEXPAND | wxALL, 5 );
    sizer->Add( subsfile_sizer, 0, wxEXPAND | wxALL, 5 );
    panel->SetSizerAndFit( sizer );
    return panel;
}

wxPanel *OpenDialog::DiscPanel( wxWindow* parent )
{
    wxPanel *panel = new wxPanel( parent, -1, wxDefaultPosition,
                                  wxSize(200, 200) );

    wxBoxSizer *sizer_row = new wxBoxSizer( wxVERTICAL );
    wxFlexGridSizer *sizer = new wxFlexGridSizer( 2, 3, 20 );

    static const wxString disc_type_array[] =
    {
        wxU(_("DVD (menus)")),
        wxU(_("DVD")),
        wxU(_("VCD")),
        wxU(_("Audio CD")),
        wxU(_("DVD (test)"))
    };

    disc_type = new wxRadioBox( panel, DiscType_Event, wxU(_("Disc type")),
                                wxDefaultPosition, wxDefaultSize,
                                WXSIZEOF(disc_type_array), disc_type_array,
                                WXSIZEOF(disc_type_array), wxRA_SPECIFY_COLS );

    sizer_row->Add( disc_type, i_disc_type_selection, wxEXPAND | wxALL, 5 );

    wxStaticText *label = new wxStaticText( panel, -1, wxU(_("Device name")) );
    disc_device = new wxTextCtrl( panel, DiscDevice_Event, wxT(""),
                                  wxDefaultPosition, wxDefaultSize,
                                  wxTE_PROCESS_ENTER);

    sizer->Add( label, 0, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    sizer->Add( disc_device, 1, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );

    disc_title_label = new wxStaticText( panel, -1, wxU(_("Title")) );
    disc_title = new wxSpinCtrl( panel, DiscTitle_Event );

    sizer->Add( disc_title_label, 0, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    sizer->Add( disc_title, 1, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );

    disc_chapter_label = new wxStaticText( panel, -1, wxU(_("Chapter")) );
    disc_chapter = new wxSpinCtrl( panel, DiscChapter_Event );
    sizer->Add( disc_chapter_label, 0,
                wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    sizer->Add( disc_chapter, 1, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    sizer_row->Add( sizer, 0, wxEXPAND | wxALL, 5 );

    panel->SetSizerAndFit( sizer_row );
    return panel;
}

wxPanel *OpenDialog::NetPanel( wxWindow* parent )
{
    int i;
    wxPanel *panel = new wxPanel( parent, -1, wxDefaultPosition,
                                  wxSize(200, 200) );

    wxBoxSizer *sizer_row = new wxBoxSizer( wxVERTICAL );
    wxFlexGridSizer *sizer = new wxFlexGridSizer( 2, 4, 20 );

    static const wxString net_type_array[] =
    {
        wxU(_("UDP/RTP")),
        wxU(_("UDP/RTP Multicast")),
        wxU(_("HTTP/FTP/MMS")),
        wxU(_("RTSP"))
    };

    for( i=0; i<4; i++ )
    {
        net_radios[i] = new wxRadioButton( panel, NetRadio1_Event + i,
                                           net_type_array[i],
                                           wxDefaultPosition, wxDefaultSize,
                                           wxRB_SINGLE );

        net_subpanels[i] = new wxPanel( panel, -1,
                                        wxDefaultPosition, wxDefaultSize );
    }

    /* UDP/RTP row */
    wxFlexGridSizer *subpanel_sizer;
    wxStaticText *label;
    i_net_ports[0] = config_GetInt( p_intf, "server-port" );
    subpanel_sizer = new wxFlexGridSizer( 3, 1, 20 );
    label = new wxStaticText( net_subpanels[0], -1, wxU(_("Port")) );
    net_ports[0] = new wxSpinCtrl( net_subpanels[0], NetPort1_Event,
                                   wxString::Format(wxT("%d"), i_net_ports[0]),
                                   wxDefaultPosition, wxDefaultSize,
                                   wxSP_ARROW_KEYS,
                                   0, 16000, i_net_ports[0] );

    subpanel_sizer->Add( label, 0, wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
    subpanel_sizer->Add( net_ports[0], 1,
                         wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    net_ipv6 = new wxCheckBox( net_subpanels[0], NetForceIPv6_Event,
                               wxU(_("Force IPv6")));
    subpanel_sizer->Add( net_ipv6, 0,
                         wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
    net_subpanels[0]->SetSizerAndFit( subpanel_sizer );
    net_radios[0]->SetValue( TRUE );

    /* UDP/RTP Multicast row */
    subpanel_sizer = new wxFlexGridSizer( 4, 1, 20 );
    label = new wxStaticText( net_subpanels[1], -1, wxU(_("Address")) );
    net_addrs[1] = new wxTextCtrl( net_subpanels[1], NetAddr2_Event, wxT(""),
                                   wxDefaultPosition, wxDefaultSize,
                                   wxTE_PROCESS_ENTER);
    subpanel_sizer->Add( label, 0, wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
    subpanel_sizer->Add( net_addrs[1], 1,
                         wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );

    label = new wxStaticText( net_subpanels[1], -1, wxU(_("Port")) );
    i_net_ports[1] = i_net_ports[0];
    net_ports[1] = new wxSpinCtrl( net_subpanels[1], NetPort2_Event,
                                   wxString::Format(wxT("%d"), i_net_ports[1]),
                                   wxDefaultPosition, wxDefaultSize,
                                   wxSP_ARROW_KEYS,
                                   0, 16000, i_net_ports[1] );

    subpanel_sizer->Add( label, 0, wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
    subpanel_sizer->Add( net_ports[1], 1,
                         wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    net_subpanels[1]->SetSizerAndFit( subpanel_sizer );

    /* HTTP and RTSP rows */
    for( i=2; i<4; i++ )
    {
        subpanel_sizer = new wxFlexGridSizer( 2, 1, 20 );
        label = new wxStaticText( net_subpanels[i], -1, wxU(_("URL")) );
        net_addrs[i] = new wxTextCtrl( net_subpanels[i], NetAddr1_Event + i,
                                       (i == 2) ? wxT("") : wxT("rtsp://"),
                                       wxDefaultPosition, wxSize( 200, -1 ),
                                       wxTE_PROCESS_ENTER);
        subpanel_sizer->Add( label, 0, wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
        subpanel_sizer->Add( net_addrs[i], 1,
                             wxEXPAND | wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
        net_subpanels[i]->SetSizerAndFit( subpanel_sizer );
    }

    /* Stuff everything into the main panel */
    for( i=0; i<4; i++ )
    {
        sizer->Add( net_radios[i], 0, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL |
                    wxALL, 5 );
        sizer->Add( net_subpanels[i], 1, wxEXPAND | wxALIGN_LEFT |
                    wxALIGN_CENTER_VERTICAL | wxALL, 5  );
    }

    sizer_row->Add( sizer, 0, wxEXPAND | wxALL, 5 );

    panel->SetSizerAndFit( sizer_row );
    return panel;
}

void OpenDialog::UpdateMRL()
{
    UpdateMRL( i_current_access_method );
}

void OpenDialog::UpdateMRL( int i_access_method )
{
    wxString demux, mrltemp;

    i_current_access_method = i_access_method;

    switch( i_access_method )
    {
    case FILE_ACCESS:
        //mrltemp = wxT("file") + demux + wxT(":") + file_combo->GetValue();
        mrltemp = file_combo->GetValue();
        break;
    case DISC_ACCESS:
      i_disc_type_selection = disc_type->GetSelection();

      switch ( i_disc_type_selection )
        {
        case 0: /* DVD with menues */
          disc_chapter->Enable();
          disc_chapter_label->Enable();
          mrltemp = wxT("dvd://")
                  + disc_device->GetValue()
                  + wxString::Format( wxT("@%d:%d"),
                                      disc_title->GetValue(),
                                      disc_chapter->GetValue() );
          break;

        case 1: /* DVD of some sort */
          disc_chapter->Enable();
          disc_chapter_label->Enable();
          mrltemp = wxT("dvdsimple://")
                  + disc_device->GetValue()
                  + wxString::Format( wxT("@%d:%d"),
                                      disc_title->GetValue(),
                                      disc_chapter->GetValue() );
          break;

        case 2:  /* VCD of some sort */
          {
            /* The chapter object is used for subtitles */

            int i_subtitle = disc_chapter->GetValue();
            config_PutInt( p_intf, "spu-channel", i_subtitle );
            
            disc_chapter->Enable();
            disc_chapter_label->Enable();
#ifdef HAVE_VCDX
          if ( disc_title->GetValue() )
            mrltemp = wxT("vcdx://")
              + disc_device->GetValue()
              + wxString::Format( wxT("@%c%d"),
                                  config_GetInt( p_intf, "vcdx-PBC"  )
                                  ? 'P' : 'E',
                                  disc_title->GetValue()
                                  );
          else
            mrltemp = wxT("vcdx://")
              + disc_device->GetValue();
#else
          mrltemp = wxT("vcd://")
            + disc_device->GetValue()
            + wxString::Format( wxT("@%d"),
                                disc_title->GetValue() );
#endif
          break;
          }
          

        case 3: /* CD-DA */
          disc_chapter->Disable();
          disc_chapter_label->Disable();
#ifdef HAVE_CDDAX
          if ( disc_title->GetValue() )
            mrltemp =  wxT("cddax://")
                  + disc_device->GetValue()
                  + wxString::Format( wxT("@T%d"),
                                      disc_title->GetValue() );
          else
            mrltemp = wxT("cddax://")
                  + disc_device->GetValue();

#else
          mrltemp =  wxT("cdda://")
                  + disc_device->GetValue()
                  + wxString::Format( wxT("@%d"),
                                      disc_title->GetValue() );
#endif
          break;

        case 4: /* DVD of some sort */
          disc_chapter->Enable();
          disc_chapter_label->Enable();
          mrltemp = wxT("dvdnav://")
                  + disc_device->GetValue()
                  + wxString::Format( wxT("@%d:%d"),
                                      disc_title->GetValue(),
                                      disc_chapter->GetValue() );
          break;

        default: ;
          msg_Err( p_intf, "invalid selection (%d)",
                   disc_type->GetSelection() );
        }

        break;
    case NET_ACCESS:
        switch( i_net_type )
        {
        case 0:
            mrltemp = wxT("udp") + demux + wxT("://");
            if ( net_ipv6->GetValue() )
            {
                mrltemp += wxT("@[::]");
            }
            if( i_net_ports[0] !=
                config_GetInt( p_intf, "server-port" ) )
            {
                mrltemp += wxString::Format( wxT("@:%d"), i_net_ports[0] );
            }
            break;

        case 1:
            mrltemp = wxT("udp") + demux + wxT("://@");
            if ((net_addrs[1]->GetLineText(0).Find (':') != -1)
                && (net_addrs[1]->GetLineText(0)[0u] != '['))
            {
                /* automatically adds '[' and ']' to IPv6 addresses */
                mrltemp += wxT("[") + net_addrs[1]->GetLineText(0)
                         + wxT("]");
            }
            else
            {
                mrltemp += net_addrs[1]->GetLineText(0);
            }
            if( i_net_ports[1] != config_GetInt( p_intf, "server-port" ) )
            {
                mrltemp += wxString::Format( wxT(":%d"), i_net_ports[1] );
            }
            break;

        case 2:
            /* http access */
            if( net_addrs[2]->GetLineText(0).Find(wxT("http://")) )
            {
                mrltemp = wxT("http") + demux + wxT("://");
            }
            mrltemp += net_addrs[2]->GetLineText(0);
            break;

        case 3:
            /* RTSP access */
            if( net_addrs[3]->GetLineText(0).Find(wxT("rtsp://")) != 0 )
            {
                mrltemp = wxT("rtsp") + demux + wxT("://");
            }
            mrltemp += net_addrs[3]->GetLineText(0);
            break;
        }
        break;

    default:
        {
            int i_item = i_access_method - MAX_ACCESS;

            if( i_item < 0 || i_item >= (int)input_tab_array.GetCount() )
                break;

            AutoBuiltPanel *input_panel = input_tab_array.Item( i_item );

            mrltemp = input_panel->name + wxT("://");

            for( int i=0; i < (int)input_panel->config_array.GetCount(); i++ )
            {
                ConfigControl *control = input_panel->config_array.Item(i);

                mrltemp += wxT(" :");

                if( control->GetType() == CONFIG_ITEM_BOOL &&
                    !control->GetIntValue() ) mrltemp += wxT("no-");

                mrltemp += control->GetName();

                switch( control->GetType() )
                {
                case CONFIG_ITEM_STRING:
                case CONFIG_ITEM_FILE:
                case CONFIG_ITEM_DIRECTORY:
                case CONFIG_ITEM_MODULE:
                    mrltemp += wxT("=\"") + control->GetPszValue() + wxT("\"");
                    break;
                case CONFIG_ITEM_INTEGER:
                    mrltemp +=
                        wxString::Format( wxT("=%i"), control->GetIntValue() );
                    break;
                case CONFIG_ITEM_FLOAT:
                    mrltemp +=
                        wxString::Format(wxT("=%f"), control->GetFloatValue());
                    break;
                }
            }

            if( input_panel->p_advanced_mrl_combo &&
                input_panel->p_advanced_mrl_combo->GetValue() )
            {
                mrltemp += wxT(" ") +
                    input_panel->p_advanced_mrl_combo->GetValue();
            }
        }
        break;
    }

    mrl_combo->SetValue( mrltemp );
}

/*****************************************************************************
 * Events methods.
 *****************************************************************************/
void OpenDialog::OnOk( wxCommandEvent& WXUNUSED(event) )
{
    mrl = SeparateEntries( mrl_combo->GetValue() );
    mrl_combo->Append( mrl_combo->GetValue() );
    if( mrl_combo->GetCount() > 10 ) mrl_combo->Delete( 0 );
    mrl_combo->SetSelection( mrl_combo->GetCount() - 1 );

    if( i_method == OPEN_STREAM )
    {
        Hide();
        if( IsModal() ) EndModal( wxID_OK );
        return;
    }

    /* Update the playlist */
    playlist_t *p_playlist =
        (playlist_t *)vlc_object_find( p_intf, VLC_OBJECT_PLAYLIST,
                                       FIND_ANYWHERE );
    if( p_playlist == NULL ) return;

    for( int i = 0; i < (int)mrl.GetCount(); i++ )
    {
        vlc_bool_t b_start = !i && i_open_arg;
        playlist_item_t *p_item =
            playlist_ItemNew( p_intf, (const char*)mrl[i].mb_str(),
                              (const char *)mrl[i].mb_str() );

        /* Insert options */
        while( i + 1 < (int)mrl.GetCount() &&
               ((const char *)mrl[i + 1].mb_str())[0] == ':' )
        {
            playlist_ItemAddOption( p_item, mrl[i + 1].mb_str() );
            i++;
        }

        /* Get the options from the subtitles dialog */
        if( subsfile_checkbox->IsChecked() && subsfile_mrl.GetCount() )
        {
            for( int j = 0; j < (int)subsfile_mrl.GetCount(); j++ )
            {
                playlist_ItemAddOption( p_item, subsfile_mrl[j].mb_str() );
            }
        }

        /* Get the options from the stream output dialog */
        if( sout_checkbox->IsChecked() && sout_mrl.GetCount() )
        {
            for( int j = 0; j < (int)sout_mrl.GetCount(); j++ )
            {
                playlist_ItemAddOption( p_item, sout_mrl[j].mb_str() );
            }
        }

        int i_id = playlist_AddItem( p_playlist, p_item,
                                     PLAYLIST_APPEND, PLAYLIST_END );

        if( b_start )
        {
            int i_pos = playlist_GetPositionById( p_playlist , i_id );
            playlist_Command( p_playlist, PLAYLIST_GOTO, i_pos );
        }
    }

    vlc_object_release( p_playlist );

    Hide();

    if( IsModal() ) EndModal( wxID_OK );
}

void OpenDialog::OnCancel( wxCommandEvent& WXUNUSED(event) )
{
    Hide();

    if( IsModal() ) EndModal( wxID_CANCEL );
}

void OpenDialog::OnPageChange( wxNotebookEvent& event )
{
    UpdateMRL( event.GetSelection() );
}

void OpenDialog::OnMRLChange( wxCommandEvent& event )
{
    //mrl = SeparateEntries( event.GetString() );
}

/*****************************************************************************
 * File panel event methods.
 *****************************************************************************/
void OpenDialog::OnFilePanelChange( wxCommandEvent& WXUNUSED(event) )
{
    UpdateMRL( FILE_ACCESS );
}

void OpenDialog::OnFileBrowse( wxCommandEvent& WXUNUSED(event) )
{
    if( file_dialog == NULL )
        file_dialog = new wxFileDialog( this, wxU(_("Open File")),
            wxT(""), wxT(""), wxT("*"), wxOPEN | wxMULTIPLE );

    if( file_dialog && file_dialog->ShowModal() == wxID_OK )
    {
        wxArrayString paths;
        wxString path;

        file_dialog->GetPaths( paths );

        for( size_t i = 0; i < paths.GetCount(); i++ )
        {
            if( paths[i].Find( wxT(' ') ) >= 0 )
                path += wxT("\"") + paths[i] + wxT("\" ");
            else
                path += paths[i] + wxT(" ");
        }

        file_combo->SetValue( path );
        file_combo->Append( path );
        if( file_combo->GetCount() > 10 ) file_combo->Delete( 0 );
        UpdateMRL( FILE_ACCESS );
    }
}

/*****************************************************************************
 * Disc panel event methods.
 *****************************************************************************/
void OpenDialog::OnDiscPanelChange( wxCommandEvent& event )
{
    UpdateMRL( DISC_ACCESS );
}

void OpenDialog::OnDiscDeviceChange( wxCommandEvent& event )
{
    char *psz_device;

    switch( disc_type->GetSelection() )
    {
        case 3:
            psz_device = config_GetPsz( p_intf, "cd-audio" );
            break;

        case 2:
            psz_device = config_GetPsz( p_intf, "vcd" );
            break;

        default:
            psz_device = config_GetPsz( p_intf, "dvd" );
            break;
    }

    if ( !psz_device ) psz_device = "";

    if( disc_device->GetValue().Cmp( wxL2U( psz_device ) ) )
    {
        b_disc_device_changed = true;
    }

    UpdateMRL( DISC_ACCESS );
}

void OpenDialog::OnDiscTypeChange( wxCommandEvent& WXUNUSED(event) )
{
    char *psz_device = NULL;
    int  i_selection = 1; /* Default Title/Track selection number*/

    switch( disc_type->GetSelection() )
    {

    case 0: /* DVD with menus */
    case 4: /* DVD with menus (dvdnav) */
        i_selection=0;
        /* Fall through... */

    case 1: /* DVD of some sort */
        psz_device = config_GetPsz( p_intf, "dvd" );
        if( !b_disc_device_changed )
        {
            if( psz_device )
                disc_device->SetValue( wxL2U(psz_device) );
            else
                disc_device->SetValue( wxT("") );

            disc_title_label->SetLabel ( wxU(_("Title")) );
        }
        disc_title->SetRange( i_selection, 255 );
        disc_title->SetValue( i_selection );

        disc_chapter->SetRange( 1, 255 );
        disc_chapter->SetValue( 1 );
        disc_chapter_label->SetLabel ( wxU(_("Chapter")) );

        break;

    case 2:  /* VCD of some sort */
        psz_device = config_GetPsz( p_intf, "vcd" );
        if( !b_disc_device_changed )
        {
            if( psz_device )
                disc_device->SetValue( wxL2U(psz_device) );
            else
                disc_device->SetValue( wxT("") );
        }

        /* There are at most 98, tracks in a VCD, 999 Segments, 500 entries
           I don't know what the limit is for LIDs, 999 is probably safe
           though.

           FIXME: it would be better however to get the information for
           this particular Media possibly from the General Info area.
         */
#ifdef HAVE_VCDX
        disc_title_label->SetLabel ( config_GetInt( p_intf, "vcdx-PBC"  )
                                     ? wxT("Playback LID") : wxT("Entry") );
        disc_title->SetRange( 0, 999 );
        i_selection = 0;
#else
        disc_title_label->SetLabel ( wxU(_("Track")) );
        disc_title->SetRange( 1, 98 );
#endif
        disc_title->SetValue( i_selection );

        /* We use the chapter to set subtitle number */
        disc_chapter_label->SetLabel ( wxU(_("Subtitle")) );
        disc_chapter->SetRange( -1, 4 );
        disc_chapter->SetValue( config_GetInt( p_intf, "spu-channel" ) );
        break;

    case 3: /* CD-DA */
        psz_device = config_GetPsz( p_intf, "cd-audio" );
        if( !b_disc_device_changed )
        {
            if( psz_device )
                disc_device->SetValue( wxL2U(psz_device) );
            else
                disc_device->SetValue( wxT("") );
        }
        disc_title_label->SetLabel ( wxU(_("Track")) );
#ifdef HAVE_CDDAX
        i_selection = 0;
#endif
       /* There are at most 99 tracks in a CD-DA */
        disc_title->SetRange( i_selection, 99 );
        disc_title->SetValue( i_selection );
        break;
    default:
        msg_Err( p_intf, "invalid Disc type selection (%d)",
                 disc_type->GetSelection() );
        break;
    }

    if( psz_device ) free( psz_device );

    UpdateMRL( DISC_ACCESS );
}

/*****************************************************************************
 * Net panel event methods.
 *****************************************************************************/
void OpenDialog::OnNetPanelChange( wxCommandEvent& event )
{
    if( event.GetId() >= NetPort1_Event && event.GetId() <= NetPort3_Event )
    {
        i_net_ports[event.GetId() - NetPort1_Event] = event.GetInt();
    }

    UpdateMRL( NET_ACCESS );
}

void OpenDialog::OnNetTypeChange( wxCommandEvent& event )
{
    int i;

    i_net_type = event.GetId() - NetRadio1_Event;

    for(i=0; i<4; i++)
    {
        net_radios[i]->SetValue( event.GetId() == (NetRadio1_Event+i) );
        net_subpanels[i]->Enable( event.GetId() == (NetRadio1_Event+i) );
    }

    UpdateMRL( NET_ACCESS );
}

/*****************************************************************************
 * Subtitles file event methods.
 *****************************************************************************/
void OpenDialog::OnSubsFileEnable( wxCommandEvent& event )
{
    subsfile_button->Enable( event.GetInt() != 0 );
}

void OpenDialog::OnSubsFileSettings( wxCommandEvent& WXUNUSED(event) )
{
    /* Show/hide the open dialog */
    if( subsfile_dialog == NULL )
        subsfile_dialog = new SubsFileDialog( p_intf, this );

    if( subsfile_dialog && subsfile_dialog->ShowModal() == wxID_OK )
    {
        subsfile_mrl.Empty();
        subsfile_mrl.Add( wxString(wxT("sub-file=")) +
                          subsfile_dialog->file_combo->GetValue() );
        if( subsfile_dialog->encoding_combo )
            subsfile_mrl.Add( wxString(wxT("subsdec-encoding=")) +
                              subsfile_dialog->encoding_combo->GetValue() );
        subsfile_mrl.Add( wxString::Format( wxT("sub-delay=%i"),
                          subsfile_dialog->delay_spinctrl->GetValue() ) );
        subsfile_mrl.Add( wxString::Format( wxT("sub-fps=%i"),
                          subsfile_dialog->fps_spinctrl->GetValue() ) );
    }
}

/*****************************************************************************
 * Stream output event methods.
 *****************************************************************************/
void OpenDialog::OnSoutEnable( wxCommandEvent& event )
{
    sout_button->Enable( event.GetInt() != 0 );
}

void OpenDialog::OnSoutSettings( wxCommandEvent& WXUNUSED(event) )
{
    /* Show/hide the open dialog */
    if( sout_dialog == NULL )
        sout_dialog = new SoutDialog( p_intf, this );

    if( sout_dialog && sout_dialog->ShowModal() == wxID_OK )
    {
        sout_mrl = sout_dialog->GetOptions();
    }
}

/*****************************************************************************
 * Utility functions.
 *****************************************************************************/
wxArrayString SeparateEntries( wxString entries )
{
    vlc_bool_t b_quotes_mode = VLC_FALSE;

    wxArrayString entries_array;
    wxString entry;

    wxStringTokenizer token( entries, wxT(" \t\r\n\""), wxTOKEN_RET_DELIMS );

    while( token.HasMoreTokens() )
    {
        entry += token.GetNextToken();

        if( entry.IsEmpty() ) continue;

        if( !b_quotes_mode && entry.Last() == wxT('\"') )
        {
            /* Enters quotes mode */
            entry.RemoveLast();
            b_quotes_mode = VLC_TRUE;
        }
        else if( b_quotes_mode && entry.Last() == wxT('\"') )
        {
            /* Finished the quotes mode */
            entry.RemoveLast();
            b_quotes_mode = VLC_FALSE;
        }
        else if( !b_quotes_mode && entry.Last() != wxT('\"') )
        {
            /* we found a non-quoted standalone string */
            if( token.HasMoreTokens() ||
                entry.Last() == wxT(' ') || entry.Last() == wxT('\t') ||
                entry.Last() == wxT('\r') || entry.Last() == wxT('\n') )
                entry.RemoveLast();
            if( !entry.IsEmpty() ) entries_array.Add( entry );
            entry.Empty();
        }
        else
        {;}
    }

    if( !entry.IsEmpty() ) entries_array.Add( entry );

    return entries_array;
}
